/**
 * Manages multi-page forms and timing.
 *
 * Author: Pierre Dragicevic
 */
 
 var startTimes = new Array();
 var endTimes = new Array();
 var DEBUG_showAllPages = false;
 var DEBUG_firstPage = 1;

////////////////////////////////////////////////////////////////////////////
// Page timing functions
////////////////////////////////////////////////////////////////////////////

/**
 * Initializes timers and page visibility. Call at page load.
 */

function initPages() {

	// For each page in the document...
	for (var page = 1; pageExists(page); page++) {
	
		// Init the start and end completion times to zero
		startTimes[page] = 0;
		endTimes[page] = 0;
		
		// Create a hidden form field that will enventually store page completion time.
		var divBlock = getPage(page);
		var timeField = document.createElement('input');
		timeField.setAttribute('type', 'hidden');
		timeField.setAttribute('id', 'timeOnPage' + page);
		timeField.setAttribute('name', 'timeOnPage' + page);
		timeField.setAttribute('value', 0);
		divBlock.appendChild(timeField);
		
		// Make the first page visible and all subsequent pages invisible.
		setPageVisible(page, page == DEBUG_firstPage);
	}
}

/**
 * Stops all timers and write their value in hidden form fields. Call when pressing the submit button.
 */
function finishPages() {

	// Finish all timings (needed for the last page).
	for (var page = 1; pageExists(page); page++) {
		if (startTimes[page] != 0 && endTimes[page] == 0) // will happen for the last page
			endTimes[page] = new Date().getTime();
	}
	
	// Write times in the hidden form fields.
	for (var page = 1; pageExists(page); page++) {
		var timeField = document.getElementById('timeOnPage' + page);
		timeField.setAttribute('value', (endTimes[page] - startTimes[page]) / 1000.0);
	}
}

/**
 * Shows the given page and hide the previous one. Call when pressing a Next Page button.
 */
function revealPage(page) {
	setPageVisible(page - 1, false);
	// This adds a flicker, otherwise the user might not realize the page has changed
	// when they are similar.
	setTimeout("setPageVisible(" + page + ", true);", 200);
	
}

////////////////////////////////////////////////////////////////////////////
// Countdown timer
////////////////////////////////////////////////////////////////////////////

var countdownStartTime = 0;
var countdownDuration = 0;
var countdownPage = 0;
var timer;

/**
 * Starts the countdown
 */
function startCountdown(page, duration, pageToShowWhenTimeElapsed) {
	countdownStartTime = new Date().getTime();
	countdownDuration = duration;
	countdownPage = page;
	updateCountdown(pageToShowWhenTimeElapsed);
	setCountdownVisible(true);
}

function endCountdown() {
    if (timer!=undefined){
	    clearTimeout(timer);
	}
}


/**
 * Makes the countdown timer visible or invisible using the element ID "countdownContainer"
 */
function setCountdownVisible(visible) {
	div = document.getElementById('countdownContainer');
	if (visible) {
		div.style.display = "block";
	} else {
		div.style.display = "none";
	}
}

/**
 * Updates the countdown timer using the element ID "countdownValue"
 */ 
function updateCountdown(pageToShowWhenTimeElapsed)
{
	div = document.getElementById('countdownValue');
	elapsed = Math.round(countdownDuration - ((new Date().getTime()) - countdownStartTime) / 1000);
	
	// finished.
	if (elapsed <= 0) {
		//div.innerHTML = 'Please proceed to the next page';
		//setAllFieldsEnabled('page' + countdownPage, false, 'next');
		showSinglePage(pageToShowWhenTimeElapsed);
		return;
	}
	
	elapsed_minutes = Math.floor(elapsed / 60);
	elapsed_seconds = elapsed - elapsed_minutes * 60;
	elapsed_string = (elapsed_seconds == 0 ? "00" : (elapsed_seconds < 10 ? "0" + elapsed_seconds : elapsed_seconds));
	if (countdownDuration >= 60)
		elapsed_string = elapsed_minutes + ":" + elapsed_string;
	div.innerHTML = elapsed_string;
	timer = setTimeout("updateCountdown('"+pageToShowWhenTimeElapsed+"')", 1000);
}

////////////////////////////////////////////////////////////////////////////
// Helper functions
////////////////////////////////////////////////////////////////////////////

// Sets a page visible or invisible using div IDs.
function setPageVisible(page, visible) {

	if (!pageExists(page))
		return;

	// Find the div block enclosing the page
	var divBlock = getPage(page);
		
	// Sets the page visible or invisible
	// and update start or end time
	if (visible) {
		if (!DEBUG_showAllPages)
			divBlock.style.display = "block";
		else
			divBlock.style.border="3px solid black";
		startTimes[page] = new Date().getTime();
	} else {
		if (!DEBUG_showAllPages)
			divBlock.style.display = "none";
		else
			divBlock.style.border = "none";
		if (startTimes[page] != 0)
			endTimes[page] = new Date().getTime();
	}
}

// Returns whether the page exists.
function pageExists(page) {
	return getPage(page);
}

// Returns the div block that encloses the given page.
function getPage(page) {
	return document.getElementById("page" + page);
}

// Returns the number of pages.
function getPageCount() {
	// For each page in the document...
	var page;
	for (page = 1; pageExists(page); page++);
	return page - 1;
}
