"""
Ellipse and related functions

Author: Luana Micallef, January 2016
"""

from shapely.geometry import Polygon
import numpy as np
import math


class Ellipse():
    """Creates ellipses and carries out related geometric functions such as finding the intersection points of 2 overlapping ellipses.
       Properties:
       - xc, yc is the centre of the ellipse 
       - axis_minor is the length of the minor axis from the centre of the ellipse to the curve 
       - axis_major is the length of the major axis from the centre of the ellipse to the curve
       - angle is the orientation of the major axis as an anticlockwise angle from the x-axis in degrees; 
               a check is carried out inside the class to ensure that this angle is in [0, 180) degrees, 
               so angle < 90 indicates a +ve gradient major axis and angle > 90 indicates a -ve gradient major axis
    """

    # Constructor
    def __init__(self, xc, yc, axis_major, axis_minor, angle):
        self.xc = xc
        self.yc = yc
        self.axis_major = self.checkAxis(axis_major)
        self.axis_minor = self.checkAxis(axis_minor)
        self.Polyline = None
        self.setAngle(angle)
    
    
    # Set the angle of the ellipse, by first ensuring that the angle is in [0,180)
    def setAngle(self, major_axis_angle_anticlockwise_deg):
    
        # ensuring that the angle is in [0,180)
        major_axis_angle_anticlockwise_deg_norm = major_axis_angle_anticlockwise_deg
        if (major_axis_angle_anticlockwise_deg < 0):
            major_axis_angle_anticlockwise_deg_norm = 180 + major_axis_angle_anticlockwise_deg
        # the following is more of a precaution as you should never have angle >=180; this ensures that the returned angle is always in [0,180)
        elif (major_axis_angle_anticlockwise_deg >= 180):
            major_axis_angle_anticlockwise_deg_norm = major_axis_angle_anticlockwise_deg - 180
        
        self.angle = major_axis_angle_anticlockwise_deg_norm


    # Set the axis in such a way that an ellipse is still an ellipse not a line when one of the axes 
    # is either 0 or nan; this occurs when the ellipse is too elongated and too similar to a line
    # e.g., when corr is 1 or -1
    def checkAxis(self, axis):
        if math.isnan(axis) or (axis==0.0):
            return 0.0000000000001
        else:
            return axis
    

    # Convert ellipse curve to a polyline
    # source: http://stackoverflow.com/questions/15445546/finding-intersection-points-of-two-ellipses-python
    def ellipseToPolyline(self, n=100):
        t = np.linspace(0, 2*np.pi, n, endpoint=False)
        st = np.sin(t)
        ct = np.cos(t)
        
        angle_radians = np.deg2rad(self.angle)
        sa = np.sin(angle_radians)
        ca = np.cos(angle_radians)
        p = np.empty((n, 2))
        p[:, 0] = self.xc + self.axis_major * ca * ct - self.axis_minor * sa * st
        p[:, 1] = self.yc + self.axis_major * sa * ct + self.axis_minor * ca * st
        
        self.Polyline = Polygon(p)
      
    
    # Determine whether a point is in the ellipse or not
    def isPointInEllipse(self,x,y):
        return ((math.pow(((x-self.xc)*math.cos(self.angle))+((y-self.yc)*math.sin(self.angle)),2)/float(math.pow(self.axis_major,2))) +
                (math.pow(((y-self.yc)*math.cos(self.angle))-((x-self.xc)*math.sin(self.angle)),2)/float(math.pow(self.axis_minor,2))) <= 1)



# Compute the area of the ellipse 
def getEllipseArea(e):
    if not e.Polyline:
        e.ellipseToPolyline()
    return e.Polyline.area
        

# Find the area of the overlap between this ellipse and another ellipse
#  (inspired from http://streamhacker.com/2010/03/23/python-point-in-polygon-shapely/)
def getApproxEllipseOverlapArea(A, B):
    if not A.Polyline:
        A.ellipseToPolyline()
    if not B.Polyline:
        B.ellipseToPolyline()
    overlapPoly = A.Polyline.intersection(B.Polyline)
    overlapArea = overlapPoly.area
    return overlapArea


# Find area of overlap between every pair for ellipses
# Input: ellipses - an array of Ellipse objects
# Output: an adjacency matrix with n rows and n columns where n is the number of ellipses and the cells are filled up with the overlap area;
#         only half of the matrix is filled up e.g., for 5 ellipses only these cells would have the overlap area 
#         [[0 1], [0 2], [0 3], [0 4], [1 2], [1 3], [1 4], [2 3], [2 4], [3 4]]; the rest are not used and filled up with 0
def getPairwiseEllipseOverlapAreas(ellipses, TriangleIndices = None):

    NumEllipses = len(ellipses)
    OverlapAreas = np.zeros((NumEllipses, NumEllipses))
    if (TriangleIndices==None):
        TriangleIndices = np.stack(np.triu_indices(NumEllipses, 1), axis=-1)
    
    for pair in TriangleIndices:
        OverlapAreas[pair[0], pair[1]] = getApproxEllipseOverlapArea(ellipses[pair[0]], ellipses[pair[1]])

    return OverlapAreas


# Filter out all the points that in the ellipse and return only the points that are not
# inputs: e = an ellipse, points = an array of cartesian points to be checked, adjustment - an ellipse size adjustment in percentage 
def findPointsNotInEllipse(e, points, adjustment=1):
    e_adjusted = e
    if (adjustment != 1):
        e_adjusted = Ellipse (e.xc, e.yc, e.axis_major*adjustment, e.axis_minor*adjustment, e.angle)

    ps_notInEllipse = []
    for x,y in points:
        if (not e_adjusted.isPointInEllipse(x,y)):
            ps_notInEllipse.append((x,y))

    return np.array(ps_notInEllipse)        
    
    