"""
The main script that runs the model-based scatterplot optimization system 

Authors: Luana Micallef, Antti Oulasvirta, Tino Weinkauf, Gregorio Palmas, August 2015
"""


import random
import time
import ellipse
import model
import utilities
import plottingintermdata  # for testing purposes only
import pp
import numpy as np
import scipy.stats
import ImageMetrics
import sys, math


# evaluating one design and returning 
# (1) whether the design should be skipped, (2) the objective score, (3) a log of all the measures, 
# (4) a list of image buffers of the clusters rendered with the current design
def evaluateDesign(design, DataBoundingBox, datapointsClusters, TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors, MaxOutliersPerc, datapoints,
                   clustersMeasures, actualCovEllipses, actualSDySDxRatios, SP, ClustersColors,
                   bSavePlotWithAxes, dataFileName, scatterplotImageFilePath, warningsFilePath, plotIntermData,
                   Weights):

    if (plotIntermData):
        runid = time.strftime("%Y%m%d_%H%M%S")
        scatterplotImageFilePath = scatterplotImageFilePath.replace(".png","_"+runid+".png")  # for TESTING only

    skipdesign = False

    marker_size, marker_opacity, image_width, image_aspect_ratio = design

    #print design,
    print "(",
    for i, d in enumerate(design):
        print "%.5g" % d,
        if i < len(design) - 1: print "\t",
    print ")",
        
    # Initiate objective score and other factors for each cluster
    objective_score = 0
    overlapFactor_foreachcluster = []
    overplottingFactor_foreachcluster = []
    lightnessFactor_foreachcluster = []
    meanFactor_foreachcluster = []
    contrastFactor_foreachcluster = []
    axis_actual_foreachcluster = []
    angle_actual_foreachcluster = []
    SDySDxRatio_actual_foreachcluster = []
    axis_perceived_foreachcluster = []
    angle_perceived_foreachcluster = []
    SDySDxRatio_perceived_foreachcluster = []
    axis_error_foreachcluster = []
    angle_error_foreachcluster = []
    unit_error_foreachcluster = []
    perceived_bbox_error_foreachcluster = []
    SDySDxRatio_actualperceived_diff_foreachcluster = []
    SDySDxRatio_error_foreachcluster = []
    direction_error_foreachcluster = []
    perceived_covellipse_foreachcluster = []
    perceived_datapoints_foreachcluster = {}
    

    BuffersClusters = {}

    # Compute objective score of each cluster
    for d in range(0,len(datapointsClusters)):
     
        # Data points in the dth index cluster
        datapointsCluster = datapointsClusters[d]
        clusterWeighting = float(len(datapointsCluster))/float(len(datapoints))
         
        # Generate the image scatterplot from the data file
        overlapFactor_ofCluster, \
        overplottingFactor_ofCluster, \
        lightnessFactor_ofCluster, \
        meanFactor_ofCluster, \
        contrastFactor_ofCluster, \
        ImageBuffer = \
                SP.GeneratePlotImageFile(datapointsCluster,
                                scatterplotImageFilePath,
                                DataBoundingBox,
                                marker_size,
                                marker_opacity,
                                image_width,
                                image_aspect_ratio,
                                bSavePlotWithAxes)


        BuffersClusters[d] = ImageBuffer

        #import skimage
        #from PIL import Image
        #Image.fromarray(skimage.img_as_ubyte(ImageBuffer)).save('out.png')

        overlapFactor_foreachcluster.append(overlapFactor_ofCluster)
        overplottingFactor_foreachcluster.append(overplottingFactor_ofCluster)
        lightnessFactor_foreachcluster.append(lightnessFactor_ofCluster)
        meanFactor_foreachcluster.append(meanFactor_ofCluster)	
        contrastFactor_foreachcluster.append(contrastFactor_ofCluster)
         
        # Data measures: ellipse properties and SDy/SDx
        actual_minorOnMajorAxis, actual_angle = clustersMeasures[d]
        actual_SDySDxRatio = actualSDySDxRatios[d]
        actual_covellipse = actualCovEllipses[d]
             
        # Get the covariance ellipse and other properties for the perceived data points
        bComputeCOVErrors = (Weights[0] != 0 or Weights[1] != 0)
        if bComputeCOVErrors:
            COVResult = model.getPerceivedCovarianceEllipse(ImageBuffer)
        else:
            COVResult = (1,np.array([]),None,0,0)
        wereEdgesPerceived, \
        perceived_datapoints, \
        perceived_covellipse, \
        perceived_minorOnMajorAxis, \
        perceived_angle = COVResult
                        
        
        # Skip design as soon as no edges are perceived in a cluster
        if not wereEdgesPerceived:
            warning = "NO PERCEIVABLE EDGES DETECTED for reordered cluster " + str(d) + ": design "+ str(design) +" was thus DISCARDED.\n"
            utilities.writeToFile(warningsFilePath, warning, append=True)
            print "\n*** WARNING: " + warning
            break
        
        
        if bComputeCOVErrors:
            # Check record of the perceived covellipses and datapoints for each cluster 
            perceived_covellipse_foreachcluster.append(perceived_covellipse)
            perceived_datapoints_foreachcluster[d] = perceived_datapoints
            
            # Get the SDy/SDx from the perceived points and the direction of the point cloud
            perceived_SDySDxRatio = model.getSDySDxRatioOfPointCloud(perceived_datapoints)

            # the axis and angle error of this current cluster (i.e., the
            # difference between the axis or angle of the perceived and actual;
            # both axis and angle are normalized to a value in [0,1])
            axis_error = clusterWeighting * abs(actual_minorOnMajorAxis - perceived_minorOnMajorAxis)
            #angle_error = clusterWeighting * model.getSmallestAngleBetweenMajorAxesOfActualAndPerceivedCovEllipses(actual_covellipse,perceived_covellipse)

            angle_error = abs(actual_covellipse.angle - perceived_covellipse.angle)
            if angle_error > 90:
                bigger_angle = max(actual_covellipse.angle, perceived_covellipse.angle)
                smaller_angle = min(actual_covellipse.angle, perceived_covellipse.angle)
                angle_error = (smaller_angle + 180) - bigger_angle
             

            angle_error = clusterWeighting * (angle_error/90.)
            unit_error = axis_error + angle_error
            SDySDxRatio_actualperceived_diff = 1 - min(actual_SDySDxRatio,perceived_SDySDxRatio)/max(actual_SDySDxRatio,perceived_SDySDxRatio)
            #SDySDxRatio_error = clusterWeighting * SDySDxRatio_actualperceived_diff
            direction_error = clusterWeighting * model.getPointCloudDirectionError(actual_covellipse,perceived_covellipse)
        
            XPerc, YPerc = perceived_datapoints.T

            rangeXPerceived = np.max(XPerc) - np.min(XPerc)
            rangeYPerceived = np.max(YPerc) - np.min(YPerc)

            bboxRatio = float(min(rangeXPerceived, rangeYPerceived)) / float(max(rangeXPerceived, rangeYPerceived))

            perceived_bbox_error = clusterWeighting * (1 - bboxRatio)


            Std_XPerceived = np.std(XPerc)
            Std_YPerceived = np.std(YPerc)
            perceived_SDySDxRatio = float(min(Std_XPerceived, Std_YPerceived)) / float(max(Std_XPerceived, Std_YPerceived))

            SDySDxRatio_error = clusterWeighting * (1 - perceived_SDySDxRatio)

        else:
            axis_error = 0
            perceived_minorOnMajorAxis = 0
            angle_error = 0
            unit_error = 0
            SDySDxRatio_actualperceived_diff = 0
            direction_error = 0
            SDySDxRatio_error = 0
            perceived_bbox_error = 0

        # Issue a warning if any of the goodness measures are NOT in [0,1]
        # - this is more of a precaution than a necessity as these measures
        # were designed to be in [0,1]
        goodnessmeasures = [("actual_minorOnMajorAxis",actual_minorOnMajorAxis), \
                            ("perceived_minorOnMajorAxis",perceived_minorOnMajorAxis), \
                            ("axis_error",axis_error), ("angle_error",angle_error), \
                            ("SDySDxRatio_actualperceived_diff",SDySDxRatio_actualperceived_diff), ("SDySDxRatio_error",SDySDxRatio_error), \
                            ("direction_error",direction_error) ]
        for measure in goodnessmeasures:
            (measureL, measureV) = measure
            if (measureV < 0 or measureV > 1):
                warning = "MEASURE NOT IN [0,1]: "+ measureL + " has value " + str(measureV) + " for design " + str(design) + "and reordered cluster " +str(d)+ ". Design was still considered (NOT discarded).\n"
                utilities.writeToFile(warningsFilePath, warning, append=True)
                print "\n*** WARNING: " + warning 
     
        # axis_actual_foreachcluster, angle_actual_foreachcluster and SDySDxRatio_actual_foreachcluster 
        # to add details to the log file
        axis_actual_foreachcluster.append(actual_minorOnMajorAxis)
        angle_actual_foreachcluster.append(actual_angle)
        SDySDxRatio_actual_foreachcluster.append(actual_SDySDxRatio)
     
        # axis_perceived_foreachcluster and angle_perceived_foreachcluster to
        # add details to the log file
        axis_perceived_foreachcluster.append(perceived_minorOnMajorAxis)
        angle_perceived_foreachcluster.append(perceived_angle)

        SDySDxRatio_perceived_foreachcluster.append(0)
     
        # axis_error_foreachcluster, angle_error_foreachcluster and unit_error_foreachcluster
        # to add details to the log file
        axis_error_foreachcluster.append(axis_error)
        angle_error_foreachcluster.append(angle_error)
        unit_error_foreachcluster.append(unit_error)
        SDySDxRatio_actualperceived_diff_foreachcluster.append(SDySDxRatio_actualperceived_diff)
        SDySDxRatio_error_foreachcluster.append(SDySDxRatio_error) 
        direction_error_foreachcluster.append(direction_error)
        perceived_bbox_error_foreachcluster.append(perceived_bbox_error)

    # Skip the entire design as soon as one cluster cannot be perceived anymore     
    if not wereEdgesPerceived:
        skipdesign = True
        return (skipdesign, 1e8, "")

    # Sum the foreachcluster values
    total_axis_error = np.sum(axis_error_foreachcluster)
    total_angle_error = np.sum(angle_error_foreachcluster)
    total_unit_error = np.sum(unit_error_foreachcluster)
    totalSDySDxRatio_error = np.sum(SDySDxRatio_error_foreachcluster)
    total_direction_error = np.sum(direction_error_foreachcluster)
    total_perceived_bbox_error = np.sum(perceived_bbox_error_foreachcluster)

    # Are the clusters perceivable when rendered together?
    if Weights[-2] != 0 or Weights[-1] != 0:
        (FinalImage, FinalAlphaMask, ClusterPerceivability) = ImageMetrics.MeasureClusterPerceivability(BuffersClusters, ClustersColors)
        # Are the outliers perceivable when rendered together with their clusters?

        if Weights[-1] != 0:
            OutliersPerceivability = ImageMetrics.MeasureOutliersPerceivability(FinalImage, \
                                                                                FinalAlphaMask, \
                                                                                TotalPointsWithoutOutliers, \
                                                                                TotalPointsWithoutOutliersColors, \
                                                                                MaxOutliersPerc, \
                                                                                [marker_size, \
                                                                                 marker_opacity, \
                                                                                 image_width, \
                                                                                 image_aspect_ratio], \
                                                                                DataBoundingBox, \
                                                                                SP)
        else:
            OutliersPerceivability = 0
    else:
        (FinalImage, FinalAlphaMask, ClusterPerceivability) = (None, None, 0)
        OutliersPerceivability = 0


    # the various image metrics have to be computed on the plot with all the
    # points in all the clusters; all factors are in [0,1] and all factors
    # should be minimized
    overlapFactor, overplottingFactor, lightnessFactor, meanFactor, contrastFactor, ImageBuffer = \
        SP.GeneratePlotImageFile(datapoints, scatterplotImageFilePath, DataBoundingBox, marker_size, marker_opacity, image_width, image_aspect_ratio, bSavePlotWithAxes)
     
    # Plotting the perceived data points and corresponding cluster variance
    # ellipses for TESTING only
    if (plotIntermData):
        plottingintermdata.plotPerceivedDataPointsAndClusterCovEllipses(perceived_datapoints_foreachcluster, perceived_covellipse_foreachcluster, dataFileName)   # for TESTING only
        #plottingintermdata.savePerceivedCovEllipseOverlapAreas(perceivedCovEllipsesOverlapAreas,


    Scores = np.array(
                        [
                            total_angle_error,
                            total_axis_error,
                            totalSDySDxRatio_error,
                            total_direction_error,
                            total_perceived_bbox_error,
                            overlapFactor,
                            overplottingFactor,
                            lightnessFactor,
                            meanFactor,
                            contrastFactor,
                            ClusterPerceivability,
                            OutliersPerceivability
                        ]
                     )

    # Final Objective Score
    objective_score = np.dot(Weights, Scores)
    
    if (plotIntermData):
        scatterplotImageFilePath = scatterplotImageFilePath.replace(".png",".txt")  # for TESTING only
        file = open(scatterplotImageFilePath, 'w')
        file.write(str(objective_score))
        file.close();

    print "%.4g" % objective_score
     
    measureslog = (','.join(str(e) for e in axis_actual_foreachcluster)) \
             +"|"+(','.join(str(e) for e in angle_actual_foreachcluster)) \
             +"|"+(','.join(str(e) for e in SDySDxRatio_actual_foreachcluster)) \
             +"|"+(','.join(str(e) for e in axis_perceived_foreachcluster)) \
             +"|"+(','.join(str(e) for e in angle_perceived_foreachcluster)) \
             +"|"+(','.join(str(e) for e in SDySDxRatio_perceived_foreachcluster)) \
             +"|"+(','.join(str(e) for e in axis_error_foreachcluster)) \
             +"|"+(','.join(str(e) for e in angle_error_foreachcluster)) \
             +"|"+(','.join(str(e) for e in unit_error_foreachcluster)) \
             +"|"+(','.join(str(e) for e in SDySDxRatio_actualperceived_diff_foreachcluster)) \
             +"|"+(','.join(str(e) for e in SDySDxRatio_error_foreachcluster)) \
             +"|"+(','.join(str(e) for e in direction_error_foreachcluster)) \
             +"|"+(','.join(str(e) for e in perceived_bbox_error_foreachcluster)) \
             +"|"+(','.join(str(e) for e in overlapFactor_foreachcluster)) \
             +"|"+(','.join(str(e) for e in overplottingFactor_foreachcluster)) \
             +"|"+(','.join(str(e) for e in lightnessFactor_foreachcluster)) \
             +"|"+(','.join(str(e) for e in meanFactor_foreachcluster)) \
             +"|"+(','.join(str(e) for e in contrastFactor_foreachcluster)) \
             +"|"+str(overlapFactor)+"|"+str(overplottingFactor) \
             +"|"+str(lightnessFactor)+"|"+str(meanFactor)+"|"+str(contrastFactor) \
             +"|"+str(ClusterPerceivability) \
             +"|"+str(OutliersPerceivability) \
             +"|"+str(objective_score)+"|"+str(len(datapointsClusters))
     
    return (skipdesign, objective_score, measureslog)


#Plots the winner to a file.        
def PlotWinner(winner_design, winner_objective_score, winningplotFilePath, runid, datapointsClusters, datapoints, ClustersColors, SP, DataBoundingBox):
    
    winningplotFilePath = winningplotFilePath.replace(".png","_"+runid+".png")
    print "Converged, printing the winner to file", winningplotFilePath
    print "Winning design:  " + str(winner_design) + "  (" + str(winner_objective_score) + ")"
    
    #creating the vector of color for each data point
    total_colors = np.zeros((len(datapoints),3))

    start_off = 0
    for cluster_idx in range(0,len(datapointsClusters)):
        total_colors[start_off:start_off + len(datapointsClusters[cluster_idx])] = ClustersColors[cluster_idx]
        start_off = start_off + len(datapointsClusters[cluster_idx])
    
    SP.GeneratePlotImageFile(datapoints, winningplotFilePath, DataBoundingBox, winner_design[0], winner_design[1], winner_design[2], winner_design[3], True,  total_colors)


#Plots the design without axes but with colors to a file.        
def PlotColoredDesign(design, FilePath, datapointsClusters, datapoints, ClustersColors, SP, DataBoundingBox):

    #creating the vector of color for each data point
    total_colors = np.zeros((len(datapoints),3))

    start_off = 0
    for cluster_idx in datapointsClusters:
        total_colors[start_off:start_off + len(datapointsClusters[cluster_idx])] = ClustersColors[cluster_idx]
        start_off = start_off + len(datapointsClusters[cluster_idx])
    
    SP.GeneratePlotImageFile(datapoints, FilePath, DataBoundingBox, design[0], design[1], design[2], design[3], True, total_colors)
    FilePath = FilePath.replace(".png","_gray.png")
    SP.GeneratePlotImageFile(datapoints, FilePath, DataBoundingBox, design[0], design[1], design[2], design[3], False)


# Evaluate designs at random 
def randomSearch(number_of_repetitions,marker_sizes,marker_opacities,image_widths,image_aspect_ratios,max_marker_size, \
                 max_number_of_designs,DataBoundingBox,datapointsClusters,ClustersColors,datapoints, \
                 outliers_foreachcluster,clustersMeasures,actualCovEllipses,actualSDySDxRatios,SP,bSavePlotWithAxes,dataFileName, \
                 scatterplotImageFilePath,winningplotFilePath,warningsFilePath,logFilePath,log_headers,plotIntermData,TriangleIndices, Weights, CreateEllipses = False):
                 
   
    logFilePath_main = logFilePath
    
    (TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors) = utilities.GetDataWithoutOutliers(datapointsClusters, outliers_foreachcluster, ClustersColors)
    
    
    (TotalOutliers, TotalOutliersColors) = utilities.MergeDataPoints(outliers_foreachcluster, ClustersColors)
    TotalColors = utilities.GetColorArray(datapointsClusters, ClustersColors)
    
    max_marker_opacity = max(marker_opacities)
    min_img_width = min(image_widths)
    min_aspect_ratio = min(image_aspect_ratios)

    OutliersDesign = [max_marker_size, max_marker_opacity, min_img_width, min_aspect_ratio]
    MaxOutlierPerc = ImageMetrics.GetMaximumOutliersPerc(TotalOutliers, TotalOutliersColors, datapoints, TotalColors, OutliersDesign, DataBoundingBox, SP)

    NumberOfDesigns = len(marker_sizes) * len(marker_opacities) * len(image_widths) * len(image_aspect_ratios)

    for r in range(number_of_repetitions): 
    
        # The current date and time in the form of: 20160106_140411
        # Data points run ID
        runid = time.strftime("%Y%m%d_%H%M%S")

        print "\n******\n"+dataFileName+" (run "+str(r)+")******\n"

        # Output image plot file names and file paths 
        if (plotIntermData):
            scatterplotImageFilePath = scatterplotImageFilePath.replace(".png","_"+runid+".png")  # for TESTING only
        logFilePath = logFilePath_main
        logFilePath = logFilePath.replace("_log.log","_"+runid+"_log.log")
        warningsFilePath = warningsFilePath.replace("_warnings.txt","_"+runid+"_warnings.txt")
        
        # logging info to log each optimized design that is found
        utilities.writeToFile(logFilePath, log_headers, append=False)
        
              
        # optimization loop
        converged = False
        n_iters_without_improvement = 0	    	# a variable to keep track of convergence
        max_iters_without_improvement = 0.01 * NumberOfDesigns     # convergence threshold
        incumbent_score = 1e8					# This is a minimization task, we start from a high value unlikely to occur
        incumbent_design = None				    # The design of the current incumbent
        designs_evaluated = dict()				# A dict of all the designs tried
        n_iters_overall = 0						# Overall number of iterations
        
        start_datetime = time.strftime("%c")    # The current date and time in the form of: Sat Oct 5 00:04:59 2015 
        
        n_incumbent = 0
        termination_reason = "Termination reason: "
        
        print "\n"+log_headers


        while not converged:
     
            # Initialize visual variables by random choice (inefficient!)
            marker_size = random.choice(marker_sizes)
            marker_opacity = random.choice(marker_opacities)
            image_width = random.choice(image_widths)
            image_aspect_ratio = random.choice(image_aspect_ratios)
        
            # Check whether the current randomly selected design has already been evaluated
            design = (marker_size, marker_opacity, image_width, image_aspect_ratio)
            if (design in designs_evaluated):
                continue
            else:
                designs_evaluated[design] = 1
            
            # evaluate the current design   
            skipdesign, objective_score, measureslog = \
                            evaluateDesign (design, DataBoundingBox, datapointsClusters, TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors, MaxOutlierPerc, \
                                            datapoints, clustersMeasures, actualCovEllipses, actualSDySDxRatios, SP, ClustersColors, bSavePlotWithAxes, dataFileName, \
                                            scatterplotImageFilePath, warningsFilePath, plotIntermData, Weights)  
                                        
            # skip the current design if necessary
            if (skipdesign):
                continue

            #print "%.4g" % objective_score
        
            # Check if the plot has been improved
            if objective_score < incumbent_score:
                incumbent_score = objective_score
                incumbent_design = design
        
                end_datetime = time.strftime("%c") 
        
                log_incumbent = str(n_incumbent) \
                        +"|"+str(n_iters_overall)+"|"+str(n_iters_without_improvement)+"|"+str(max_iters_without_improvement) \
                        +"|"+str(start_datetime)+"|"+str(end_datetime) \
                        +"|"+str(max_marker_size) \
                        +"|"+"|".join(str(x) for x in design) \
                        +"|"+measureslog+"\n"
            
                utilities.writeToFile(logFilePath, log_incumbent, append=True)
                print "Incumbent:   " + log_incumbent
            
                # move incumbent plot to winner dir
                #os.rename(scatterplotImageFilePath, winningplotFilePath.replace(".png","_incumbent"+str(n_incumbent)+".png"))
        
                n_iters_without_improvement = 0
                n_incumbent += 1
                start_datetime = time.strftime("%c") 

            else:
                n_iters_without_improvement += 1
                if n_iters_without_improvement == max_iters_without_improvement:
                    converged = True
                    termination_reason = termination_reason + "n_iters_without_improvement == max_iters_without_improvement"
                
            if (len(designs_evaluated) >= max_number_of_designs):
                converged = True	
                termination_reason = termination_reason + "no more designs to evaluate"
                
            n_iters_overall += 1
            
        # Finalization: Print winner to file
        PlotWinner(incumbent_design, incumbent_score, winningplotFilePath, runid, datapointsClusters, datapoints, ClustersColors, SP, DataBoundingBox)
        utilities.writeToFile(logFilePath, termination_reason, append=True)

        if CreateEllipses:
            print "creating ellipses for the winning design"
            utilities.CreateEllipseImages(DataBoundingBox, incumbent_design, datapointsClusters, ClustersColors, winningplotFilePath.replace(".png","_"+runid+"_ell.png"))



# Evaluate designs by doing an exhaustive search
#ClusterColors is the vector of the colors of each point in the clusters, array of RGB values of size n
def exhaustiveSearch(allDesigns,max_marker_size,max_number_of_designs,DataBoundingBox,datapointsClusters,ClustersColors,datapoints, \
                     outliers_foreachcluster, clustersMeasures,actualCovEllipses,actualSDySDxRatios,SP,bSavePlotWithAxes,dataFileName,scatterplotImageFilePath,winningplotFilePath, \
                     warningsFilePath,logFilePath,log_headers,plotIntermData,TriangleIndices, Weights, CreateEllipses = False):
    
    
    # The current date and time in the form of: 20160106_140411
    runid = time.strftime("%Y%m%d_%H%M%S")
    
    # Add run id to the output image plot file names and file paths to know when the optimization was run
    #if (plotIntermData):
    #    scatterplotImageFilePath = scatterplotImageFilePath.replace(".png","_"+runid+".png")  # for TESTING only
    winningplotFilePath = winningplotFilePath.replace(".png","_"+runid+".png")
    logFilePath = logFilePath.replace("_log.log","_"+runid+"_log.log")
    warningsFilePath = warningsFilePath.replace("_warnings.txt","_"+runid+"_warnings.txt")
    
    # logging info to log each optimized design that is found
    utilities.writeToFile(logFilePath, log_headers, append=False)
        
    
    # optimization loop
    converged = False
    incumbent_score = 1e8					# This is a minimization task, we start from a high value unlikely to occur
    incumbent_design = None				    # The design of the current incumbent
    n_iters_overall = 0						# Overall number of iterations
    
    start_datetime = time.strftime("%c")    # The current date and time in the form of: Sat Oct 5 00:04:59 2015 
    
    n_incumbent = 0
    termination_reason = "Termination reason: "
    
    print "\n"+log_headers
    
    (TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors) = utilities.GetDataWithoutOutliers(datapointsClusters, outliers_foreachcluster, ClustersColors)
    
    
    (TotalOutliers, TotalOutliersColors) = utilities.MergeDataPoints(outliers_foreachcluster, ClustersColors)
    TotalColors = utilities.GetColorArray(datapointsClusters, ClustersColors)
    
    npAllDesigns = np.array(allDesigns)

    max_marker_opacity = npAllDesigns.max(axis = 0)[1]
    min_img_width = npAllDesigns.min(axis = 0)[2]
    min_aspect_ratio = npAllDesigns.min(axis = 0)[3]

    OutliesrsDesign = [max_marker_size, max_marker_opacity, min_img_width, min_aspect_ratio]
    MaxOutlierPerc = ImageMetrics.GetMaximumOutliersPerc(TotalOutliers, TotalOutliersColors, datapoints, TotalColors, OutliesrsDesign, DataBoundingBox, SP)

    winnerLine = ""

    while not converged:
        
        # current design
        design = allDesigns[n_iters_overall]
        
        # evaluate the current design   
        skipdesign, objective_score, measureslog = \
                        evaluateDesign (design, DataBoundingBox, datapointsClusters, TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors, MaxOutlierPerc, \
                                        datapoints, clustersMeasures, actualCovEllipses, actualSDySDxRatios, SP, ClustersColors,  bSavePlotWithAxes, dataFileName, \
                                        scatterplotImageFilePath, warningsFilePath, plotIntermData, Weights)  
                                    
        # skip the current design if necessary
        if (skipdesign):
            n_iters_overall += 1
            if (n_iters_overall >= max_number_of_designs):
                break
            else:
                continue

        #print "%.4g" % objective_score
        start_datetime = time.strftime("%c") 
        end_datetime = time.strftime("%c") 

        log_design = str(n_incumbent) \
                    +"|"+str(n_iters_overall)+"|"+str(max_number_of_designs) \
                    +"|"+str(start_datetime)+"|"+str(end_datetime) \
                    +"|"+str(max_marker_size) \
                    +"|"+"|".join(str(x) for x in design) \
                    +"|"+measureslog+"\n"
        
        utilities.writeToFile(logFilePath, log_design, append=True)

        # Check if the plot has been improved
        if objective_score < incumbent_score:
            incumbent_score = objective_score
            incumbent_design = design
    
            
        
            # move incumbent plot to winner dir
            #os.rename(scatterplotImageFilePath, winningplotFilePath.replace(".png","_incumbent"+str(n_incumbent)+".png"))
    
            n_incumbent += 1
           

            winnerLine = log_design
        
        
        #print "Incumbent:   " + log_design

        n_iters_overall += 1
        if (n_iters_overall >= max_number_of_designs):
            converged = True	
            termination_reason = termination_reason + "no more designs to evaluate"
            
    if (incumbent_design!=None):
        # Finalization: Print winner to file
        PlotWinner(incumbent_design, incumbent_score, winningplotFilePath, runid, datapointsClusters, datapoints, ClustersColors, SP, DataBoundingBox)
        utilities.writeToFile(logFilePath, winnerLine, append=True)
        utilities.writeToFile(logFilePath, termination_reason, append=True)


        if CreateEllipses:
            print "creating ellipses for the winning design"
            utilities.CreateEllipseImages(DataBoundingBox, incumbent_design, datapointsClusters, ClustersColors, winningplotFilePath.replace(".png","_"+runid+"_ell.png"))





# Evaluate designs by doing an exhaustive search
#ClusterColors is the vector of the colors of each point in the clusters, array of RGB values of size n
def exhaustiveSearch_PP (allDesigns,max_marker_size,max_number_of_designs,DataBoundingBox,datapointsClusters, ClustersColors, datapoints, \
                         outliers_foreachcluster,clustersMeasures,actualCovEllipses,actualSDySDxRatios,SP,bSavePlotWithAxes,dataFileName,scatterplotImageFilePath,winningplotFilePath, \
                         warningsFilePath,logFilePath,log_headers,plotIntermData,TriangleIndices, Weights, CreateEllipses = False):
    
    
    # The current date and time in the form of: 20160106_140411
    runid = time.strftime("%Y%m%d_%H%M%S")
    
    # Add run id to the output image plot file names and file paths to know when the optimization was run
    if (plotIntermData):
        scatterplotImageFilePath = scatterplotImageFilePath.replace(".png","_"+runid+".png")  # for TESTING only
    logFilePath = logFilePath.replace("_log.log","_"+runid+"_log.log")
    warningsFilePath = warningsFilePath.replace("_warnings.txt","_"+runid+"_warnings.txt")
    

    (TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors) = utilities.GetDataWithoutOutliers(datapointsClusters, outliers_foreachcluster, ClustersColors)
    
    
    (TotalOutliers, TotalOutliersColors) = utilities.MergeDataPoints(outliers_foreachcluster, ClustersColors)
    TotalColors = utilities.GetColorArray(datapointsClusters, ClustersColors)
    
    npAllDesigns = np.array(allDesigns)

    max_marker_opacity = npAllDesigns.max(axis = 0)[1]
    min_img_width = npAllDesigns.min(axis = 0)[2]
    min_aspect_ratio = npAllDesigns.min(axis = 0)[3]

    OutliesrsDesign = [max_marker_size, max_marker_opacity, min_img_width, min_aspect_ratio]
    MaxOutlierPerc = ImageMetrics.GetMaximumOutliersPerc(TotalOutliers, TotalOutliersColors, datapoints, TotalColors, OutliesrsDesign, DataBoundingBox, SP)
    
    # A very large objective score assigned to a design that should be skipped 
    largeObjScoreToSkipDesign = 1e8
    
    # tuple of all parallel python servers to connect with
    ppservers = ()
    
    # Creates jobserver with automatically detected number of workers
    # job_server = pp.Server(ppservers=ppservers)
    
    # Creates jobserver with 4 workers, assuming a 4 core machine
    job_server = pp.Server(ncpus=8)
    
    # Start time
    start_time = time.time()
    start_datetime = time.strftime("%c") 
    
    # Submitting all the jobs (one per design) and then retrieving the results that is the evaluation for each one
    jobs = [(design, job_server.submit(evaluateDesign, \
                                       (design, DataBoundingBox, datapointsClusters, TotalPointsWithoutOutliers, TotalPointsWithoutOutliersColors, MaxOutlierPerc,\
                                        datapoints, clustersMeasures, actualCovEllipses, actualSDySDxRatios, SP, ClustersColors, bSavePlotWithAxes, dataFileName,\
                                        scatterplotImageFilePath, warningsFilePath, plotIntermData, Weights,), \
                                        (), \
                                        ("ellipse","model","utilities","plottingintermdata","ImageMetrics","numpy as np","math",))) \
            for design in allDesigns]
    
    # An array with all the results
    # e.g., [(design tuple, job object), ...]
    # ... job object contains (skipdesign, objective_score, measureslog, BufferClusters) when decoded using job()
    all_results = np.array(np.array(jobs).T[1]) 

    # Get the objective score for each design
    def getObjectiveScore(job):
        skipdesign, objective_score, measureslog = job()
        if (skipdesign):
            objective_score = largeObjScoreToSkipDesign	
        return objective_score
    
    getObjectiveScores = np.vectorize(getObjectiveScore)
    objective_scores = getObjectiveScores(all_results)
    
    # Find the design with lowest objective score
    indexOfLowestObjScore = np.argmin(objective_scores)
    
    # Get design and measureslog corresponding to the lowest objective score
    winner_design, winner_job = jobs[indexOfLowestObjScore]
    winner_skipdesign, winner_objective_score, winner_measureslog = winner_job()
    
    elapsedTime = time.time() - start_time

    print "Time elapsed: ", elapsedTime, "s"
    job_server.print_stats()
    
    # Finalization: Print winner to file
    if not winner_skipdesign:
        PlotWinner(winner_design, winner_objective_score, winningplotFilePath, runid, datapointsClusters, datapoints, ClustersColors, SP, DataBoundingBox)
        
        if CreateEllipses:
            print "creating ellipses for the winning design"
            utilities.CreateEllipseImages(DataBoundingBox, winner_design, datapointsClusters, ClustersColors, winningplotFilePath.replace(".png","_"+runid+"_ell.png"))
    
    # Details of all designs to be written to log file
    end_datetime = time.strftime("%c") 
    
    log = log_headers.replace("n_incumbent|","description|")
    i=0
    for design, job in jobs:
        skipdesign, objective_score, measureslog = job()
        desc = "evaluated_design"
        if (skipdesign):
            desc = "skipped_design"
        log = log + desc \
                    +"|"+str(i)+"|"+str(max_number_of_designs) \
                    +"|"+str(start_datetime)+"|"+str(end_datetime) \
                    +"|"+str(max_marker_size) \
                    +"|"+"|".join(str(x) for x in design) \
                    +"|"+measureslog+"\n"
        i=i+1
    
    # Details of winner design to be written to log file
    if not winner_skipdesign:
        log = log + "winner_design" \
                    +"|"+str(indexOfLowestObjScore)+"|"+str(max_number_of_designs) \
                    +"|"+str(start_datetime)+"|"+str(end_datetime) \
                    +"|"+str(max_marker_size) \
                    +"|"+"|".join(str(x) for x in winner_design) \
                    +"|"+winner_measureslog+"\n"
        
        # Termination reason (and job_server.print_stats()) to be written to file log file
        termination_reason = "winner = design with lowest objective score found using an exhaustive search with parallelpython (pp)"
        log = log + termination_reason
    
    # Write log to log file
    utilities.writeToFile(logFilePath, log, append=False)
    


    outLogTimings = logFilePath.replace(".log","_timing.log")

    fileTiming = open(outLogTimings, "w")
    fileTiming.write("design per minute: "+str(float(len(allDesigns))/(elapsedTime/60.))+"\n")
    fileTiming.close()

    # Destroy jobserver otherwise PP will complain that there are "Too many open files"
    job_server.destroy() 
  
    
