import time
import multiprocessing as mp

import cv2
import numpy as np


class Vision(mp.Process):
    def __init__(self, input_queue, output_queue):
        super().__init__()
        self.input_queue = input_queue
        self.output_queue = output_queue
        self.duration = 2000  # ms

    def _millis(self):
        '''return time in milliseconds'''
        # return int(time.perf_counter()*1000)
        return int(time.time()*1000)

    def _get_readings(self, show=False):
        cap = cv2.VideoCapture(0)
        N = 100
        visual_signal = np.zeros((N, 2))
        index = 0
        time_passed = 0
        start_time = self._millis()
        while time_passed < self.duration and index < N:
            time_passed = self._millis()-start_time
            _, frame = cap.read()  # frame format: (x, y, (bgr))
            _, intensity = cv2.threshold(frame[:, :, 0], 240, 255,
                                         cv2.THRESH_BINARY)
            visual_signal[index, 0] = time_passed
            # visual_signal[index, 1] = int(np.sum(intensity) > 130000)
            visual_signal[index, 1] = np.sum(intensity)
            if show:
                cv2.imshow('frame', intensity)
                if cv2.waitKey(1) & 0xFF == ord('q'):
                    break
            index += 1
        cap.release()
        if show:
            cv2.destroyAllWindows()

        # normalize
        visual_signal[:, 1] = visual_signal[:, 1] / np.linalg.norm(
                visual_signal[:, 1], ord=np.inf)

        return visual_signal

    def run(self):
        # Threaded running
        while True:
            if not self.input_queue.empty():
                message = self.input_queue.get()
                if message == 'start':
                    self.output_queue.put(self._get_readings())
                if message == 'end':
                    break


if __name__ == '__main__':
    input_q = mp.Queue()
    output_q = mp.Queue()
    vision_process = Vision(input_q, output_q)
    vision_process.start()
    input_q.put("start")
    while output_q.empty():
        print("waiting...")
        time.sleep(1)
    result = output_q.get()
    input_q.put("end")
    for i, j in result:
        print("%6d %d" % (i, j))
