import struct
import time

import serial
import numpy as np
import matplotlib.pyplot as plt


class RobotError(Exception):
    def __init__(self, message):
        super().__init__(message)
        self.message = message


class RobotFinger:

    MIN_POSITION = 0
    MAX_POSITION = 1023

    # actions
    SEND_SIGNAL = 2
    SETUP = 6
    RANDOM_ACTION = 1
    DO_NOTHING = 3
    TEST_CONNECTION = 7
    SET_HIGHT = 3

    def __init__(self, baudrate=115200, reset=False):
        self.baudrate = 115200

        if reset:
            port = self._get_port()
            self.ser = serial.Serial(port=port, baudrate=1200)
            self.ser.close()
            time.sleep(5)

        port = self._get_port()
        self.ser = serial.Serial(port=port, baudrate=baudrate,
                                 timeout=3, write_timeout=3)
        self.check_connection("initializing")

    def close(self):
        self.ser.close()

    def reset(self):
        self.ser.close()
        port = self._get_port()
        self.ser = serial.Serial(port=port, baudrate=1200)
        self.ser.close()
        time.sleep(3)
        self.ser = serial.Serial(port=port, baudrate=self.baudrate,
                                 timeout=3, write_timeout=3)
        self.check_connection("resetting")

    def _get_response_string(self):
        '''Get response string until linebreak'''
        response = self.ser.readline().decode("utf-8")
        # print("response: {}".format(response))
        return response

    def _get_response_nstring(self, n, message):
        '''Get n chars and return them in form of string'''
        response = b''
        for i in range(n):
            self._wait_for_reply(1, 1, message)
            response += self.ser.read(1)
        return response.decode('utf-8')

    def _get_response_list(self):
        '''Get response as list of numbers'''
        response = []
        response_string = self._get_response_string()
        for item in response_string.split(','):
            response.append(int(item.strip()))
        return response

    def _get_response_number(self):
        '''Get response as a integer'''
        byte_buffer = self.ser.read(2)
        return struct.unpack('h', byte_buffer)[0]

    def _get_response_float(self):
        '''Get response as a float'''
        byte_buffer = self.ser.read(4)
        return struct.unpack('f', byte_buffer)[0]

    def _get_port(self):
        '''Try 20 ports in decreasing order. Return first valid'''
        ports = ['COM%s' % i for i in range(20, 0, -1)]
        for port in ports:
            try:
                s = serial.Serial(port)
                s.close()
                return port
            except (OSError, serial.SerialException) as e:
                pass
        raise RobotError("Could not find port!!")

    def _get_2_response_numbers(self):
        '''Get 2 response as a numbers'''
        byte_buffer = self.ser.read(4)
        return struct.unpack('hh', byte_buffer)

    def _is_position_valid(self, position):
        return self.MIN_POSITION <= position and position <= self.MAX_POSITION

    def _wait_for_reply(self, target_bytes, timeout, message):
        '''Busy wait for reply. Throws error upon timeout'''
        start = time.time()
        while self.ser.in_waiting < target_bytes:
            if time.time()-start > timeout:
                self.close()
                raise RobotError("Timeout reached while: %s" % message)


    def send_signal(self, time_step, signal):
        self.check_connection("sending signal")
        m, n = signal.shape
        self.ser.write(struct.pack('<bhh', self.SEND_SIGNAL, time_step, n))
        for i in range(n):
            for j in range(m):
                self.ser.write(struct.pack('<f', signal[j][i]))
            self._wait_for_reply(1, 3, 'writing signal')
            self.ser.read(1)

        # debugging
        # return self._get_response_string()

        # collect resulting sensor information
        # np.set_printoptions(precision=4, suppress=True)
        output = np.zeros((n, 3))
        for i in range(n):
            for j in range(3):
                self._wait_for_reply(4, 10, "reading output")
                output[i][j] = struct.unpack('f', self.ser.read(4))[0]
            self.ser.write(struct.pack('<h', 1));
        self._wait_for_reply(2, 3, 'reading button int')
        button_n = self._get_response_number()
        self._wait_for_reply(4, 3, 'reading force float')
        peak_force = self._get_response_float()
        return output, button_n, peak_force

    def do_nothing(self):
        '''Returns True (bool) once the action is complete'''
        self.ser.write(struct.pack('<bhh', self.DO_NOTHING, 0, 0))
        return bool(self._get_response_number())

    def set_hight(self, angle):
        '''Returns True (bool) once the action is complete'''
        self.ser.write(struct.pack('<bf', self.SET_HIGHT, angle))
        return self._get_response_number()

    def check_connection(self, message):
        '''Returns True (bool) once the action is complete'''
        self.ser.write(struct.pack('<b', self.TEST_CONNECTION))
        response = self._get_response_nstring(len("ready!"), message)
        if not response.startswith("Ready"):
            self.close()
            raise RobotError("Failed to form connection")

    def setup(self):
        '''
        Sets robot finger in setup mode where torque from the servomotors is
        released. Please turn the robot to its upper limit and then press a
        button to record limits of upper finger position and servomotors. Then
        move the robot to lower position and press a button to record limits in
        lower position.

        method will return a tuble raw value limits in form:
        (upper_servo_max, lower_servo_max, finger_max,
         upper_servo_min, lower_servo_min, finger_min)
        '''
        self.ser.write(struct.pack('<b', self.SETUP))
        while self.ser.in_waiting < 12:
            time.sleep(0.1)
        byte_buffer = self.ser.read(2 * 6)
        return struct.unpack('hhhhhh', byte_buffer)

    def do_random_action(self, duration):
        # duration of random robot movement in seconds!
        self.ser.write(struct.pack('<bh', self.RANDOM_ACTION, duration))
        while self.ser.in_waiting < 1:
            time.sleep(0.1)
        byte_buffer = self.ser.read(1)
        return_value =  struct.unpack('<b', byte_buffer)[0]
        return bool(return_value)

    def buffer_state(self):
        return self.ser.in_waiting
