import numpy as np
from scipy import signal
import matplotlib.pyplot as plt


NEUROMUSCULAR_NOISE = 5e-2
NOISE_CONSTANT = 0.4
N = 120


def make_square_pulse(t, offset, amplitude, duration):
    y = np.zeros(t.shape)
    for i, value in enumerate(t):
        if value <= offset or (offset+duration) < value:
            y[i] = 0
        else:
            y[i] = amplitude
    return y


def filter_data(t, data):
    fn = 4
    damping_factor = 0.9
    nat_freq = 2*np.pi*fn
    num = [nat_freq**2]
    den = [1, 2*damping_factor*nat_freq, nat_freq**2]
    tf = signal.TransferFunction(num, den)
    tout, y, x = signal.lsim(tf, data, t)
    return y


def add_noise(amplitude, signal_dependent=True):
    if signal_dependent:
        # white noise with zero mean and variance k*amplitude^2 [1]
        noise = np.random.normal(scale=(NOISE_CONSTANT*amplitude**2)**2)
    else:
        noise = np.random.normal(scale=NEUROMUSCULAR_NOISE)
    new_amplitude = min(1.0, max(0.0, amplitude+noise))
    # print("Noise: %.3g --> %.3g, diff %.3g" % (
    #     amplitude, new_amplitude, new_amplitude - amplitude))
    return new_amplitude


def make_signal(time_step, total_duration, start_delay, signal_delay,
                amplitude1, duration1, amplitude2, duration2, show=False):

    n = np.ceil(total_duration/time_step)
    if (n > N):
        raise Exception(
                "Too long signal. Lower time step or shorten duration! %d > %d"
                % (n, N))
    amplitude1 = add_noise(amplitude1)
    amplitude2 = add_noise(amplitude2)
    t = np.arange(time_step, total_duration+time_step, time_step)
    n = len(t)

    activation_upper_p = make_square_pulse(
            t, start_delay + signal_delay, amplitude1, duration1)
    activation_upper = filter_data(t/1000, activation_upper_p)

    activation_lower_p = make_square_pulse(
            t, start_delay, amplitude2, duration2)
    activation_lower = filter_data(t/1000, activation_lower_p)

    signal = np.zeros((2, n))
    # print("Samples: %d" % n)
    signal[0] = activation_upper
    signal[1] = activation_lower
    if show:
        plt.plot(t, activation_lower)
        plt.plot(t, activation_upper)
        plt.legend(['agonist\no: %d, a: %.2f d: %d' % (
            start_delay, amplitude2, duration2),
            'antagonist\no: %d, a: %.2f d: %d' % (
            start_delay + signal_delay, amplitude1, duration1)])
        plt.title('Activation signals')
        plt.xlabel("Time (ms)")
        plt.show()
    return signal


if __name__ == '__main__':
    # make pulse signal with offset, amplitude and duration
    offset = 100
    duration = 200
    amplitude = 0.8
    amplitude = add_noise(amplitude, True)
    print('amplitude %f' % amplitude)
    T = offset + duration
    sampling_period = 10
    n_samples = int((T+1000)/sampling_period)
    print('samples: %d' % n_samples)
    t = np.linspace(0, T+1000, num=n_samples)
    pulse = make_square_pulse(t, offset, amplitude, duration)
    y = filter_data(t/1000, pulse)
    plt.plot(t, pulse)
    plt.title("filter testing...")
    plt.plot(t, y)
    plt.show()



# [1] Harris, Christopher M., and Daniel M. Wolpert. "Signal-dependent noise
# determines motor planning." Nature 394.6695 (1998): 780.
