﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace TemporalPointingTestDriver
{


    public partial class Form1 : Form
    {

        // readCount: count length of data from serial port
        private volatile int readCount = 0;
        // stores data from the serial
        private volatile UInt32 lastTimestamp = UInt32.MaxValue;
        private volatile UInt16 lastData = 0;

        // buffer for logging. if a successful trial, dump the logs in the buffer to a log file.
        private List<LogData> buffer;

        // zeroPointIndex: for monitoring when the LED was turned on
        private int zeroPointIndex = -1;
        // for compesating hall sensor value drift
        private double zeroOffset = 0;

        // internal variables for decide successful button press with a trial.
        private bool isPressed = false;
        private int pressCount = 0;  // pressCount = # of trials where the button was pressed
        private int skipCount = 0;  // skipCount = # of trials without a button press after starting
        private UInt16 pressThreshold = 600;    // value threshold for deciding a button press
        private TextWriter logWriter = null;    // logger for raw log
        private TextWriter metricWriter = null; // logger for calculated metrics.
        private int distance, width, latency;   // global value for the inputted values

        private Measures measuring = null;

        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            // enumerate serial ports available
            string[] ports = SerialPort.GetPortNames();
            foreach (string port in ports)
            {
                comboBoxSerialList.Items.Add(port);
            }

            comboBoxSerialList.SelectedIndex = comboBoxSerialList.Items.Count - 1;
            buffer = new List<LogData>();

            listBoxConditions.Items.AddRange(Condition.generateConditions(true).ToArray());
            listBoxConditions.SelectedIndex = 0;

            comboBoxCondition.SelectedIndex = 0;
        }

        // close the opend serial ports
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            serialPortTester.Close();
            if (logWriter != null)
            {
                logWriter.Flush();
                logWriter.Close();
            }
        }

        private void buttonConnection_Click(object sender, EventArgs e)
        {
            tryConnect(10);
        }

        /// <summary>
        /// try to connect the serial port "tryCount" times
        /// </summary>
        /// <param name="tryCount">trial counter</param>
        private void tryConnect(int tryCount)
        {
            if (tryCount == 0)
                return;

            // Connect to serial port
            string port = (string)comboBoxSerialList.SelectedItem;
            serialPortTester.BaudRate = 115200;
            serialPortTester.PortName = port;
            serialPortTester.RtsEnable = true;
            serialPortTester.DtrEnable = true;
            serialPortTester.ReadBufferSize = 1024 * 1024;
            serialPortTester.Open();

            serialPortTester.WriteLine("b");

            // Serial Sync, wait until at least 50 dataframes are received.
            serialPortTester.ReadExisting();
            while (serialPortTester.BytesToRead < 8 * 50) ;
            //Console.WriteLine(serialPortTester.BytesToRead);

            int syncCount = 0;
            int failCount = 0;

            while (serialPortTester.BytesToRead > 8)
            {
                if (syncCount == 0)
                {
                    if (checkSync(serialPortTester))
                        syncCount++;
                }
                else
                {
                    // read exisiting databytes (6 bytes)
                    for (int i = 0; i < 6; i++)
                    {
                        serialPortTester.ReadByte();
                    }

                    if (checkSync(serialPortTester))
                    {
                        syncCount++;
                    }
                    else
                    {
                        syncCount = 0;
                        failCount++;
                    }
                }

                if (failCount > 50)
                {
                    Console.WriteLine("too many failed syncs");
                    break;
                }
            }
            //Console.WriteLine(serialPortTester.BytesToRead);

            if (syncCount >= 5)
            {
                buttonConnection.Enabled = false;
                stripLabel.Text = String.Format("Port {0} is synced.", serialPortTester.PortName);
                serialPortTester.Write("r");
                readData(serialPortTester);

                timer1.Enabled = true;
                serialPortTester.DataReceived += serialPortTester_DataReceived;
                //buttonLogStart.Enabled = true;
                buttonSet.Enabled = true;
                serialPortTester.WriteLine("s");
            }
            else
            {
                stripLabel.Text = String.Format("Port {0} is failed to sync.", serialPortTester.PortName);
                serialPortTester.Close();
                tryConnect(tryCount - 1);
            }
        }

        private bool checkSync(SerialPort sp)
        {
            if (!sp.IsOpen || sp.BytesToRead < 2)
                return false;

            byte b1 = (byte)sp.ReadByte();
            byte b2 = (byte)sp.ReadByte();

            if (b1 == 0xff && b2 == 0xf0)    // Sync character #1
            {
                return true;
            }

            return false;
        }

        private Tuple<bool[], UInt32, UInt16> readData(SerialPort sp)
        {
            if (!sp.IsOpen || sp.BytesToRead < 6)
                return null;

            byte[] timestamp = new byte[4];
            byte[] data = new byte[2];

            try
            {
                sp.Read(data, 0, 2);
                sp.Read(timestamp, 0, 4);
            }
            catch (Exception) { }
            
            bool[] info = new bool[4];

            // retrieve 4-bit additional info
            info[0] = (data[1] & 0x80) > 0;
            info[1] = (data[1] & 0x40) > 0;
            info[2] = (data[1] & 0x20) > 0;
            info[3] = (data[1] & 0x10) > 0;

            // trim out additional info;
            data[1] = (byte)(data[1] & 0x0F);

            UInt16 data_converted = BitConverter.ToUInt16(data, 0);
            UInt32 timestamp_converted = BitConverter.ToUInt32(timestamp, 0);

            return new Tuple<bool[], UInt32, UInt16>(info, timestamp_converted, data_converted);
        }


        void serialPortTester_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            SerialPort sp = (SerialPort)sender;

            if (!sp.IsOpen || sp.BytesToRead < 8)
                return;

            while (sp.IsOpen && sp.BytesToRead >= 8)
            {
                bool synced = checkSync(sp);
                if (synced)
                {
                    Tuple<bool[], UInt32, UInt16> data = readData(sp);

                    if (data == null)
                        return;
                   
                    readCount++;

                    // new dataframe. flush the buffer
                    if (lastTimestamp > data.Item2)
                    {
                        // write the dataframe only if the key was pressed.
                        if (isPressed)
                        {
                            if(logWriter != null && pressCount > 0)                            
                            {
                                int trialNo = pressCount + skipCount;
                                string writeBuffer = "";
                                double zeroPointTimestamp = distance / 2 - width / 2;

                                for (int i = 0; i < buffer.Count; i++)
                                {
                                    if(buffer[i] != null)
                                    {
                                        buffer[i].sequence = (zeroPointIndex < 0) ? -1 : i - zeroPointIndex + 1;
                                        buffer[i].trial = trialNo;
                                        buffer[i].timestamp = buffer[i].timestamp - zeroPointTimestamp;

                                        writeBuffer += buffer[i] + "\n";
                                    }
                                }
                                logWriter.Write(writeBuffer);
                            }
                                
                            //measuring.FinishMeasure();
                            //if (metricWriter != null)
                                //metricWriter.WriteLine(measuring);
                        }
                        else
                        {
                            if(pressCount > 0)
                                skipCount++;
                        }

                        // reset measures
                        if (measuring == null)
                        {
                            //measuring = new Measures(0, distance, width);
                        }
                        else
                        {
                            //measuring = new Measures(measuring.activationTime, distance, width);
                        }

                        buffer.Clear();
                        zeroPointIndex = -1;
                        isPressed = false;
                    }

                    bool LED = data.Item1[0];
                    bool feedback = data.Item1[1];
                    lastTimestamp = data.Item2;
                    lastData = data.Item3;

                    //if (measuring != null)
                       // measuring.feedData(data);
                    
                    // physical button measure
                    bool isActivated = false;
                    bool haveData = false;

                    double convertedVal = 0;

                    String buttonType = getButtonType();
                    String actuationType = getActuationType();

                    if (buttonType.Equals("p")) // physical button
                    {
                        double mmVal = Measures.ValToMM(lastData);

                        if(zeroOffset == -1)
                        {
                            zeroOffset = mmVal;
                        }

                        convertedVal = mmVal - zeroOffset;
                        if (convertedVal < 0) convertedVal = 0;

                        if (convertedVal > 0.01)
                            haveData = true;
                        if (convertedVal > 1)
                            isActivated = true;
                    }
                    else if (buttonType.Equals("t")) // touch button
                    {
                        convertedVal = lastData;
                        if(convertedVal > 0)
                        {
                            haveData = true;
                        }
                        if(convertedVal > 20)
                        {
                            isActivated = true;
                        }
                    }
                    
                    // only acquire activated dataframe
                    if (!isPressed && isActivated)
                    {
                        pressCount++;
                        isPressed = true;
                        labelCount.BeginInvoke((Action)delegate()
                        {
                            labelCount.Text = "Count = " + pressCount;
                        });
                    }

                    
                    foreach (bool b in data.Item1)
                        if (b == true) haveData = true;

                    if((lastTimestamp / 100.0) >= distance-10)
                    {
                        int reversePos = getReversePos();
                        int stopPos = getStopPos();
                        
                        if (reversePos > 0)
                        {
                            if (pressCount == reversePos && serialPortTester.IsOpen && buttonReverse.UseVisualStyleBackColor == true)
                            {
                                serialPortTester.WriteLine("r1");
                                buttonReverse.UseVisualStyleBackColor = false;
                            }
                        }
                        if(stopPos > 0)
                        {
                            if (pressCount > stopPos && serialPortTester.IsOpen)
                            {
                                serialPortTester.WriteLine("s"); // stop

                                if (textBoxName.Enabled == false)
                                {
                                    buttonLogStart.BeginInvoke((Action)delegate()
                                    {
                                        stopLogging();
                                    });
                
                                }
                            }
                        }
                    }
                    
                    if (logWriter != null)
                    {
                        if(haveData)
                        {
                            buffer.Add(new LogData(LED, feedback, (double)lastTimestamp / 100.0, convertedVal, lastData, -1, -1));
                        }
                        else
                        {
                           buffer.Add(null);
                        }

                        if(zeroPointIndex == -1 && LED == true)
                        {
                            zeroPointIndex = buffer.Count();
                        }                        
                    }
                }
                else
                {
                    stripLabel.Text = String.Format("Port {0} has been out of sync.", serialPortTester.PortName);
                    serialPortTester.Close();
                }
            }
        }

        private void timer1_Tick(object sender, EventArgs e)
        {
            if (serialPortTester.IsOpen)
            {
                labelDistance.BeginInvoke((Action)delegate() 
                {
                    stripLabel.Text = string.Format("FrameC = {0}, data={1}", (readCount)*2, lastData);
                    lastData = 0;

                    if (logWriter != null)
                    {
                        logWriter.Flush();
                    }
                    readCount = 0; 
                
                });
            }
        }

        private void buttonSet_Click(object sender, EventArgs e)
        {
            buttonSet.Enabled = false;

            bool isDistanceNumeric = int.TryParse(textBoxDistance.Text, out distance);
            bool isWidthNumeric = int.TryParse(textBoxWidth.Text, out width);
            bool isLatencyNumeric = int.TryParse(textBoxLatency.Text, out latency);

            bool isValid = true;

            if (!isDistanceNumeric || distance < 0)
            {
                textBoxDistance.BackColor = Color.Gold;
                isValid = false;
            }
            if (!isWidthNumeric || width < 0)
            {
                textBoxWidth.BackColor = Color.Gold;
                isValid = false;
            }
            if (!isLatencyNumeric || latency < 0)
            {
                textBoxLatency.BackColor = Color.Gold;
                isValid = false;
            }


            if(isValid)
            {
                textBoxDistance.BackColor = Color.White;
                textBoxWidth.BackColor = Color.White;
                textBoxLatency.BackColor = Color.White;
            
                labelDistance.Text = distance.ToString();
                labelWidth.Text = width.ToString();
                labelLatency.Text = latency.ToString();

                if (serialPortTester.IsOpen)
                {
                    serialPortTester.WriteLine("d" + distance);
                    serialPortTester.WriteLine("w" + width);
                    serialPortTester.WriteLine("l" + latency);
                    serialPortTester.WriteLine(getButtonType());
                    serialPortTester.WriteLine(getActuationType());
                    serialPortTester.WriteLine("r0"); // do not reverse the signal

                    buttonLogStart.Enabled = true;
                    buttonReverse.UseVisualStyleBackColor = true;
                }
            }

            if (radioButtonPhysical.Checked)
            {
                zeroOffset = -1;
            }

        }


        private String getButtonType()
        {
            string buttonType = "";

            if (radioButtonPhysical.Checked)
                buttonType = "p";
            else if (radioButtonTouch.Checked)
                buttonType = "t";

            return buttonType;
        }


        private int getReversePos()
        {
            int reversePos = 0;
            if (int.TryParse(textBoxReversePosition.Text, out reversePos) && reversePos > 0)
            {
                return reversePos;
            }

            return 0;
        }


        private int getStopPos()
        {
            int stopPos = 0;
            if (int.TryParse(textBoxStopPosition.Text, out stopPos) && stopPos > 0)
            {
                return stopPos;
            }
            
            return 0;
        }

        private String getActuationType()
        {
            string actuationType = "";

            if (radioButtonTraditional.Checked)
                actuationType = "a1";
            else if (radioButtonMax.Checked)
                actuationType = "a2";

            return actuationType;
        }


        private void startLogging()
        {
            String prefix = textBoxName.Text
                                + "-" + labelDistance.Text
                                + "_" + labelWidth.Text
                                + "_" + labelLatency.Text
                                + "_" + getButtonType()
                                + "_" + getActuationType()
                                + "_r" + getReversePos();

            string filename = prefix + "-" + getTimeString() + ".csv";
            string filenameMetric = prefix + "-" + getTimeString() + ".metric.csv";

            logWriter = new StreamWriter(filename, false);
            //metricWriter = new StreamWriter(filenameMetric, false);

            //logWriter.WriteLine(getConditionInformation());
            logWriter.WriteLine(LogData.GetTitleRow());

            //metricWriter.WriteLine(getConditionInformation());
            //metricWriter.WriteLine(Measures.GetTitleRow());

            if (serialPortTester.IsOpen)
            {
                serialPortTester.WriteLine("b"); // begin command
            }

            textBoxName.Enabled = false;
            buttonSet.Enabled = false;
            buttonNextCondition.Enabled = false;
            pressCount = 0;
            skipCount = 0;
            zeroPointIndex = -1;
            labelCount.Text = "Count = " + pressCount;
        }
        private void stopLogging()
        {
            // prevent null object calling
            if (logWriter != null)
            {
                TextWriter temp = logWriter;
                logWriter = null;
                temp.Flush();
                temp.Close();
            }
            if (metricWriter != null)
            {
                TextWriter temp = metricWriter;
                metricWriter = null;
                temp.Flush();
                temp.Close();
            }

            if (serialPortTester.IsOpen)
            {
                serialPortTester.WriteLine("s"); // stop command
            }
            buttonNextCondition.Enabled = true;
            textBoxName.Enabled = true;
            buttonSet.Enabled = true;
        }

        private void buttonLogStart_Click(object sender, EventArgs e)
        {
            if (textBoxName.Enabled == true)
            {
                startLogging();
            }
            else
            {
                stopLogging();
            }
        }

        private String getConditionInformation()
        {
            String ret = "";
            ret += "Name," + textBoxName.Text;
            ret += "\nDistance," + labelDistance.Text;
            ret += "\nWidth," + labelWidth.Text;
            ret += "\nbuttonType," + getButtonType();
            ret += "\nActuation," + getActuationType();
            
            ret += "\n";
            return ret;
        }

        static string getTimeString()
        {
            DateTime now = DateTime.Now;

            string time = now.ToString("yyyy_MM_dd_HH_mm_ss");
            return time;
        }

        private void conditions_ValueChanged(object sender, EventArgs e)
        {
            buttonSet.Enabled = true;
        }

        private void buttonReverse_Click(object sender, EventArgs e)
        {
            bool isReversed = !buttonReverse.UseVisualStyleBackColor;

            if (serialPortTester.IsOpen)
            {
                if(isReversed)
                {
                    serialPortTester.WriteLine("r0");
                }
                else
                {
                    serialPortTester.WriteLine("r1");
                }

                buttonReverse.UseVisualStyleBackColor = isReversed;
            }

            buttonSet.Enabled = true;
        }

        private void buttonStart_Click(object sender, EventArgs e)
        {
            if (serialPortTester.IsOpen)
            {
                serialPortTester.WriteLine("b"); // begin command
            }
        }

        private void buttonStop_Click(object sender, EventArgs e)
        {
            if (serialPortTester.IsOpen)
            {
                serialPortTester.WriteLine("s"); // begin command
            }
        }

        private void buttonNextCondition_Click(object sender, EventArgs e)
        {
            if(listBoxConditions.SelectedItem.GetType() != typeof(Condition))
            {
                return;
            }

            Condition c = (Condition)listBoxConditions.SelectedItem;
            labelActuatorType.Text = c.button.ToString(); 

            c.setCondition(serialPortTester);
            labelDistance.Text = textBoxDistance.Text = c.distance.ToString();
            labelWidth.Text = textBoxWidth.Text = c.width.ToString();
            labelLatency.Text = textBoxLatency.Text = "0";

            distance = c.distance;
            width = c.width;
            latency = 0;

            switch (c.actuation)
            {
                case Condition.actuationType.TRADITIONAL:
                    radioButtonTraditional.Checked = true; break;
                case Condition.actuationType.MAX:
                    radioButtonMax.Checked = true; break;
            }

            switch(c.button)
            {
                case Condition.buttonType.PHYSICAL:
                    radioButtonPhysical.Checked = true; break;
                case Condition.buttonType.TOUCH:
                    radioButtonTouch.Checked = true; break;
            }

            startLogging();

            if (listBoxConditions.SelectedIndex == listBoxConditions.Items.Count - 1)
            {
                buttonNextCondition.Enabled = false;
                return;
            }
            listBoxConditions.SelectedIndex++;
        }

        private void buttonPractice_Click(object sender, EventArgs e)
        {
            listBoxConditions.Items.Clear();
            listBoxConditions.Items.AddRange(Condition.generateConditions(false, true, comboBoxCondition.SelectedIndex + 1).ToArray());
            listBoxConditions.Items.Add("");
            listBoxConditions.SelectedIndex = 0;
            buttonNextCondition.Enabled = true;
        }

        private void buttonTest_Click(object sender, EventArgs e)
        {
            listBoxConditions.Items.Clear();
            listBoxConditions.Items.AddRange(Condition.generateConditions(false, false, comboBoxCondition.SelectedIndex + 1).ToArray());
            listBoxConditions.Items.Add("");
            listBoxConditions.SelectedIndex = 0;
            buttonNextCondition.Enabled = true;
        }

        private void buttonBaseline_Click(object sender, EventArgs e)
        {
            listBoxConditions.Items.Clear();
            listBoxConditions.Items.AddRange(Condition.generateConditions(true).ToArray());
            listBoxConditions.Items.Add("");
            listBoxConditions.SelectedIndex = 0;
            buttonNextCondition.Enabled = true;
        }
    }

    class LogData
    {
        public bool LED, feedback;
        public double timestamp;
        public double Val;
        public ushort rawVal;
        public int trial;
        public int sequence;

        public LogData(bool LED, bool feedback, double timestamp, double val, ushort rawVal, int trial, int sequence)
        {
            this.LED = LED;
            this.feedback = feedback;
            this.timestamp = timestamp;
            this.Val = val;
            this.rawVal = rawVal;
            this.trial = trial;
            this.sequence = sequence;
        }
        
        public static string GetTitleRow()
        {
            return "LEDstate,Feedback,Timestamp,Value,rawValue,trial,sequence";
        }

        public override string ToString()
        {
            return String.Format("{0},{1},{2},{3},{4},{5},{6}", LED ? 1 : 0, feedback ? 1 : 0, timestamp, Val, rawVal, trial, sequence);
        }
    }

    class Measures
    {
        // interesting points
        public UInt32 activationTime = 0;
        public UInt32 lightOnTime = 0;
        public UInt32 lightOffTime = 0;
        public UInt32 buttonDownTime = 0;
        public UInt32 buttonUpTime = 0;
        // calculated measures
        public long interval = 0;
        public bool isSuccess = false;
        public long shift = 0;

        private UInt32 lastActivation = 0;
        private long distance, width;
        private bool lastLEDstate = false;
        private bool lastButtonstate = false;
        private bool finalized = false;

        public Measures(UInt32 lastActivatedTime, int distance, int width)
        {
            this.lastActivation = lastActivatedTime;
            this.distance = distance*10;
            this.width = width;
        }

        public void feedData(Tuple<bool[], UInt32, UInt16> data) // collect data and record the interesting points
        {
            if (finalized)
                return;

            UInt32 timestamp = data.Item2;
            UInt16 value = data.Item3;

            bool LED = data.Item1[0];
            bool isActivated = data.Item1[1];

            // button activated! (occur only once)
            if (isActivated && this.activationTime == 0)
            {
                this.activationTime = timestamp;
            }

            // light on (rising edge)
            if (lastLEDstate == false && LED == true && this.lightOnTime == 0)
            {
                this.lightOnTime = timestamp;
            }

            // light off (falling edge)
            if (lastLEDstate == true && LED == false && this.lightOffTime == 0)
            {
                this.lightOffTime = timestamp;
            }
            
            lastLEDstate = LED;
        }        
        
        public void FinishMeasure()      // calculate the desired measures
        {
            if (lastActivation != 0 && activationTime != 0)
            {
                this.interval = (distance - lastActivation) + activationTime;
            }
            this.isSuccess = (activationTime >= lightOnTime && activationTime <= lightOffTime);
            this.shift = (long)activationTime - lightOnTime;

            finalized = true;
        }

        public static string GetTitleRow()
        {
            return "Activation,LightOn,LightOff,ButtonDown,ButtonUp,Interval,isSuccess,timeShift";
        }

        public override string ToString() 
        {
            String ret = "Not Finalized Yet";
            if (finalized)
            {
                ret = string.Format("{0},{1},{2},{3},{4},{5},{6},{7}", activationTime, lightOnTime, lightOffTime, buttonDownTime, buttonUpTime, interval, isSuccess, shift);
            }
            return ret;
        }  

        #region Value-MM Conversion related functions

        // value table for interpolation
        private static double[,] valueTable = new double[,] {
            {	590	,	0	},
            {	592	,	0.1	},
            {	594	,	0.2	},
            {	596.5	,	0.3	},
            {	599	,	0.4	},
            {	601.5	,	0.5	},
            {	604	,	0.6	},
            {	606.5	,	0.7	},
            {	609	,	0.8	},
            {	612	,	0.9	},
            {	615	,	1	},
            {	618	,	1.1	},
            {	621	,	1.2	},
            {	624	,	1.3	},
            {	627	,	1.4	},
            {	630.5	,	1.5	},
            {	634	,	1.6	},
            {	638	,	1.7	},
            {	642	,	1.8	},
            {	646.5	,	1.9	},
            {	651	,	2	},
            {	656	,	2.1	},
            {	661	,	2.2	},
            {	666.5	,	2.3	},
            {	672	,	2.4	},
            {	678	,	2.5	},
            {	684.5	,	2.6	},
            {	691	,	2.7	},
            {	698	,	2.8	},
            {	705	,	2.9	},
            {	712	,	3	},
            {	720	,	3.1	},
            {	728	,	3.2	},
            {	736	,	3.3	},
            {	745	,	3.4	},
            {	755	,	3.5	},
            {	765	,	3.6	},
            {	775.5	,	3.7	},
            {	786.5	,	3.8	},
            {	798	,	3.9	},
            {	809.5	,	4	},
            {	821	,	4.1	},
            {	833	,	4.2	},
            {	846	,	4.3	},
            {	860	,	4.4	},
        };

        // valueTable을 참조하여 센서값-mm값 상호 변환.
        public static double ValToMM(double val)
        {
            int count = valueTable.Length / 2;

            if (val < valueTable[0, 0]) // less than minimum
                return valueTable[0, 1];

            for (int i = 0; i < count - 1; i++)
            {
                double val1 = valueTable[i, 0];
                double val2 = valueTable[i + 1, 0];
                double mm1 = valueTable[i, 1];
                double mm2 = valueTable[i + 1, 1];

                // if a value lies  val1 < val < val2, then..
                if (val >= val1 && val < val2)
                {
                    double ratio = (val - val1) / (val2 - val1);
                    double mm = (mm2 - mm1) * ratio + mm1;
                    return mm;
                }
            }

            return valueTable[count - 1, 1]; // larger than maximum
        }

        public static double MMToVal(double mm)
        {
            int count = valueTable.Length / 2;

            if (mm < valueTable[0, 1]) // less than minimum
                return valueTable[0, 0];

            for (int i = 0; i < count - 1; i++)
            {
                double val1 = valueTable[i, 0];
                double val2 = valueTable[i + 1, 0];
                double mm1 = valueTable[i, 1];
                double mm2 = valueTable[i + 1, 1];

                // if a value lies  val1 < val < val2, then..
                if (mm >= mm1 && mm < mm2)
                {
                    double ratio = (mm - mm1) / (mm2 - mm1);
                    double val = (val2 - val1) * ratio + val1;
                    return val;
                }
            }

            return valueTable[count - 1, 0]; // larger than maximum
        }

        #endregion
    }

}

