/*
	Actuation point - button press signal acquisition

	      This device consists of two LEDs, one Hall sensor, and a contact detector, and a touch sensor (MPR121)
        With the dedicated PC-side program (ActuationPointTestDriver), it collects a button signals aligned to
        a given cue signal (both light and sound)

        This program keeps timestamps (1 unit = 0.1 ms) to be exactly matched with
        lighting signals and amount of button press from the hall sensor.

        // INPUT Pins: ANALOG_IN(A0), IRQ(D3)
        // OUTPUT Pins: LED1(D10), LED2(D9), DEBUGLED(D13), PIEZO(D5), SIGPIEZO(D6)

          Hall sensor: SS49E hall sensor
          - Datasheet: https://dscl.lcsr.jhu.edu/main/images/3/31/SS49e_Hall_Sensor_Datasheet.pdf
          - 5V / GND / #A0 = SIGNAL
          Touch sensor: AdaFruit MPR121
          - Datasheet: https://learn.adafruit.com/adafruit-mpr121-12-key-capacitive-touch-sensor-breakout-tutorial/overview
          - #D3 = SCL / #D2 = SDA / #D4 = IRQ
          LED 1: Confirmation LED, #D10 (PB6-OCA1B-Timer2), On at HIGH (5v) output.
          LED 2: Signal LED, #D9 (PB5-OC1A-Timer1), On at HIGH (5v) output.
          PIEZO SIGNAL  : #D5, On when LED2 is on
          PIEZO FEEDBACK: #D6, On when a button press is successful (within the given temporal width).

        ############### COMMANDS ################ (via Serial baud 115200)
        d[INT]\n          set temporal distance (ms)
        w[INT]\n          set temporal width (ms)
        l[INT]\n          set system latency (ms)
        p[D/U][INT]\n     set actuation point (hall-sensor value)
          e.g.) pD650\n  => set actuation point at value 650 when key goes down.
          e.g.) pU650\n  => set actuation point at value 650 when key goes up.
        f[0/1/2]            EnableSound(2)/Enable(1)/Disable(0) the success feedback light
        s[INT]\n          set signal lighting profile
          e.g.) s3\n     => set Profile.3 (triangle) lighting.
          Profile.0 = Constant  _|''''''''''''|_
          Profile.1 = Increase  __,...-----'''|_
          Profile.2 = Decrease  _|'''-----....__
          Profile.3 = Triangle  _...---''---..._
          Profile.4 = Impulse   _------''------_  ( 25% pre-lighting, 30ms impulse)


	Last modified day 2017 Jan. 25
	By Sunjun Kim (kuaa.net@gmail.com)
*/

#include <Wire.h>
#include "Adafruit_MPR121.h"

// You can have up to 4 on one i2c bus but one is enough for testing!
Adafruit_MPR121 cap = Adafruit_MPR121();

// MACROS

// INPUT Pins: ANALOG_IN, IRQ
// Pin #A0 (hall sensor)
#define ANALOG_IN 0
// digital 3 ==> touch sensor IRQ
#define IRQ 4

// OUTPUT Pins: LED1, LED2, DEBUGLED, PIEZO, SIGPIEZO
// Macros for fast digital IO
// digital 10 => PB6 / LED 1 : Confirm LED
#define LED1 10
#define LED1H PORTB |= 0B01000000;
#define LED1L PORTB &=~0B01000000;
// digital  9 => PB5 / LED 2: Signal LED
#define LED2 9
#define LED2H PORTB |= 0B00100000;
#define LED2L PORTB &=~0B00100000;
// digital 13 => PC7, debugLED
#define DEBUGLED 13
#define DEBUGH PORTC |= 0B10000000;
#define DEBUGL PORTC &=~0B10000000;

// digital 5, 6 => Piezo Speaker
#define PIEZO 5  // for piezo feedback buzzer
#define SIGPIEZO 6 // for piezo signal buzzer

// USER definable variables
// unit : 0.01 ms  e.g., 100,000 = 1000.0 ms = 1 sec
unsigned long distance = 100000;
unsigned long width = 2000;
unsigned long latency = 0;

unsigned long feedbackDuration = 50 * 100UL;
//int point = 612; // around 2 mm
//int shape = 0;


// =========================================== internal variables below. not for a user.
int val = 0; // analog read value from the hall sensor.

unsigned long lastMicro; // for calculate timestamp. Timestamp is an ellapsed time from "lastMicro"
// time keeping variables (will be automatically set according to distance and width)
unsigned long middleT = 0; // distance / 2
unsigned long startT = 0;  // middle - width/2
unsigned long endT = 0;    // middle + width/2

// Keeps track of the last pins touched
// so we know when buttons are 'released'
uint16_t lasttouched = 0;
uint16_t currtouched = 0;

char buttonType = 't';  // p for physical, t for touch.
int actuationType = 1;  // 1 for traditional, 2 for max.
int reverse = 0;        // 0: do not reverse the stimuli/feedback pins    1: reverse the pins

unsigned long activatedTime = 0;
bool hysteresis = false;
bool isSuccess = false;
bool previousOnFeedback = false;

bool started = false;

int NAVG = 3;
double movingAVG = 0;
int maximum = 0;

//bool on = false;
//bool falsePress = false;

void reCalculateNumbers()
{
  middleT = distance / 2;
  startT = middleT - width / 2;
  endT = middleT + width / 2;
}

// s: start position
// e: end position
// val: current position
// returns: 0.0 ~ 1.0, 0.0 when val == s, 1.0 when val == e.
// returns: -1.0 when out of range.
float getPositionRatio(unsigned long _s, unsigned long _e, unsigned long _val)
{
  if (_val < _s || _val > _e)
    return -1.0;
  unsigned long denominator = _e - _s;
  unsigned long numerator = _val - _s;

  float ratio = (float)numerator / denominator;

  return ratio;
}

void setup() {
  while (!Serial);        // needed to keep leonardo/micro from starting too fast!

  Serial.begin(115200);
  lastMicro = micros();

  // OUTPUT Pins: LED1(D10), LED2(D9), DEBUGLED(D13), PIEZO(D5), SIGPIEZO(D6)
  pinMode(LED1, OUTPUT);
  pinMode(LED2, OUTPUT);
  pinMode(DEBUGLED, OUTPUT);
  pinMode(PIEZO, OUTPUT);
  pinMode(SIGPIEZO, OUTPUT);

  digitalWrite(LED1, LOW);
  digitalWrite(LED2, LOW);
  digitalWrite(DEBUGLED, LOW);
  digitalWrite(PIEZO, LOW);
  digitalWrite(SIGPIEZO, LOW);

  // INPUT Pins: ANALOG_IN(A0), IRQ(D3)
  pinMode(IRQ, INPUT);

  // setup Fast PWM for timer 1 (pin 9). Refer Atmega32u4 datasheet, search TCCR1A / TCCR1B
  // http://www.atmel.com/Images/Atmel-7766-8-bit-AVR-ATmega16U4-32U4_Datasheet.pdf
  // http://playground.arduino.cc/Main/TimerPWMCheatsheet
  // TCCR1B: Timer/Counter1 Control Register B
  //  Setting 	Divisor 	Frequency
  //  0b001	1 	 	31372.55  <-- Selected for this sketch
  //  0b010	8 	 	3921.16
  //  0b011	64 	 	490.20   <--DEFAULT
  //  0b100	256 	 	122.55
  //  0b101 	1024 	 	30.64
  TCCR1B = (TCCR1B & 0b11111000) | 0x01;

  reCalculateNumbers();

  // Default address is 0x5A, if tied to 3.3V its 0x5B
  // If tied to SDA its 0x5C and if SCL then 0x5D
  if (!cap.begin(0x5A)) {
    //Serial.println("MPR121 not found, check wiring?");
    while (1);
  }
  //Serial.println("MPR121 found!");

  cap.writeRegister(MPR121_CONFIG1, 0x3F); // modified, 63uA charge current (maximal)

  cap.writeRegister(MPR121_GPIOEN, 0xFF);   // enable GPIO mode for all avilable pins
  cap.writeRegister(MPR121_GPIODIR, 0x00);  // set all to inputs.


  cap.writeRegister(MPR121_MHDF, 0x01); // Max half data (do not use 0)
  cap.writeRegister(MPR121_MHDR, 0x01); // Determines the largest magnitude of variation to pass through the baseline filter. The range of the effective value is 1~63.

  cap.writeRegister(MPR121_NHDF, 0x01); // Noise half data (do not use 0)
  cap.writeRegister(MPR121_NHDR, 0x05); // Determines the incremental change when non-noise drift is detected. The range of the effective value is 1~63.

  cap.writeRegister(MPR121_NCLF, 0x0E); // Noise count limit for fall/raise
  cap.writeRegister(MPR121_NCLR, 0x0E); // Determines the number of samples consecutively greater than the Max Half Delta value.
  // This is necessary to determine that it is not noise. The range of the effective value is 0~255.

  cap.writeRegister(MPR121_FDLF, 0x00); // Filter Delay Count
  cap.writeRegister(MPR121_FDLR, 0x00); // Determines the operation rate of the filter. A larger count limit means the filter delay is operating
  // more slowly. The range of the effective value is 0~255.

  cap.writeRegister(MPR121_ECR, 0x81);  // start with first 5 bits of baseline tracking, enable only electorde 0

  // setThreshholds(uint8_t touch, uint8_t release)
  cap.setThreshholds(6, 3);
}

void loop() {  
  //######################## Serial event processing
  if (Serial.available() > 0)
  {
    // Read whatever serial events here
    char c = Serial.read();
    switch (c)
    {
      case 'd': // temporal distance
        distance = readNumber() * 100UL;
        reCalculateNumbers();
        break;
      case 'w': // temporal width
        width = readNumber() * 100UL;
        reCalculateNumbers();
        break;
      case 'l': // latency
        latency = readNumber() * 1000UL;
        break;
      case 't': // signal type
        if (Serial.read() == '\n')
          buttonType = 't';
        break;
      case 'p': // signal type
        if (Serial.read() == '\n')
          buttonType = 'p';
        break;
      case 'a': // activation point type
        actuationType = readNumber();
        break;

      case 'b':
        started = true;
        break;
      case 's':
        started = false;
        activatedTime = 0;
        digitalWrite(LED1, LOW);
        digitalWrite(LED2, LOW);
        digitalWrite(DEBUGLED, LOW);
        digitalWrite(PIEZO, LOW);
        digitalWrite(SIGPIEZO, LOW);
        break;
    }
  }
  
  

  //######################## Timer and sensor value acquision
  unsigned long newMicro = micros();
  unsigned long timestamp = (newMicro - lastMicro) / 10UL;

  // signal acquisition

  // for touch sensor (MPR121)
  currtouched = cap.touched();
  bool isTouched = currtouched & _BV(0);
  bool wasTouched = lasttouched & _BV(0);

  bool upTrigger = false;
  bool downTrigger = false;

  if (isTouched && !wasTouched) {
    upTrigger = true;
  }

  if (!isTouched && wasTouched) {
    downTrigger = true;
  }
  // reset our state
  lasttouched = currtouched;

  if(!started)
    return;

  long filtered = cap.filteredData(0);
  long baseline = cap.baselineData(0);

  // for analog sensor (hall effect sensor)
  int lastVal = val;

  switch (buttonType)
  {
    case 'p': // physical button
      val = analogRead(ANALOG_IN);
      break;
    case 't': // touch button
      val = baseline - filtered;
      if (val < 0)
        val = 0;
      break;
  }

  movingAVG = ((double)(NAVG - 1) * movingAVG + val) / NAVG;
  if(val > maximum)
    maximum = val;


  // activation point calculation
  bool isActivating = false;
  if (activatedTime == 0 && !hysteresis)
  {
    switch (actuationType)
    {
      case 1:  // traditional
        if (buttonType == 'p' && val > 651 && val - movingAVG > 3) // value at 2mm = 651, when value goes up
        {
          isActivating = true;
        }
        else if (buttonType == 't' && val > 6 && val - movingAVG > 3) // almost instant after a signal acquisition.
        {
          isActivating = true;
        }
        break;
      case 2: // max.
        if (buttonType == 'p' && val > 650 && abs(val - movingAVG) < 3 && val < maximum - 1) // a button is pressed beyond some point, and signal starts to decreasing.
        {
          isActivating = true;
        }
        if (buttonType == 't' && val > 20 && abs(val - movingAVG) < 3 && val < maximum - 1)
        {
          isActivating = true;
        }
        break;
    }
  }
  if (isActivating)
  {
    activatedTime = newMicro + latency;
    hysteresis = true;
  }

  if (hysteresis)
  {
    if ((buttonType == 'p' && val < 630) || (buttonType == 't' && val < 10))
    {
      hysteresis = false;
      maximum = 0;
    }
  }

  // Debug led, blinking.
  if ((newMicro / 1000UL) % 500 < 100)  {
    DEBUGH;
  }
  else  {
    DEBUGL;
  }

  // Timer reset
  if (timestamp > distance)
  {
    lastMicro = newMicro;
    lastVal = 0;
    isSuccess = false;    
  }

  // debounce for 100 ms
  unsigned long activatedDuration = newMicro - activatedTime;
  if (activatedDuration > distance/2 * 10UL && activatedDuration < 10000000000UL) // debounce for 100 ms and preventing overflow from negative value on an unsigned type.
    activatedTime = 0;

  //######################## LED visualization
  bool isOnTime = (startT <= timestamp && timestamp <= endT);
  bool isOnFeedback = (activatedTime > 0 && 0 <= activatedDuration && activatedDuration <= feedbackDuration * 10UL); // fires only within a feedback duration

  bool isOnFeedbackTrigger = previousOnFeedback == false && isOnFeedback == true;

  previousOnFeedback = isOnFeedback;

  if (isOnTime)
  {
    //float ratio = getPositionRatio(startT, endT, timestamp);
    analogWrite(LED2, 255);
    analogWrite(SIGPIEZO, 10);

    if(isOnFeedbackTrigger) // feedback only on success
    {
      isSuccess = true;
    }
  }
  else
  {
    analogWrite(SIGPIEZO, 0);
    analogWrite(LED2, 0);  
  }

  //if(isOnFeedback && isSuccess)
  if(isOnFeedback)
  {
    analogWrite(LED1, 255);
    analogWrite(PIEZO, 10);   
  }
  else
  {
    analogWrite(LED1, 0);
    analogWrite(PIEZO, 0);
    isSuccess = false;    
  }
  

  //######################## Serial data transmission
  // add 4-bit additional info to the value
  // (2byte = 0123__012 34567890 / using 4-bit info + 10-bit value)
  // info bit 0: true if the signal lighting is on
  // info bit 1: true if the key is really activated on system (once, may be delayed by the latency)
  // info bit 2: true if the key goes down beyond the actuation point (multiple time, not affected by the latency)
  // info bit 3: reserved
  val += isOnTime * 0x8000; // info bit #0
  val += isOnFeedback * 0x4000;   // info bit #1
  val += false * 0x2000;   // info bit #2
  val += false * 0x1000;   // info bit #3

  Serial.write( 0xff );
  Serial.write( 0xf0 );

  // Send Data (2 byte), little-endian encoding.
  Serial.write( val & 0xff );
  Serial.write( (val >> 8) & 0xff );

  // Send Timestamp (4 byte), little-endian encoding.
  Serial.write((timestamp) & 0xff);
  Serial.write((timestamp >> 8) & 0xff);
  Serial.write((timestamp >> 16) & 0xff);
  Serial.write((timestamp >> 24) & 0xff);

//  Serial.print("ab ");
//  Serial.println(val);
}

int readNumber()
{
  int num = Serial.parseInt();
  if (Serial.read() == '\n')
  {
    return num;
  }
  return 0;
}


