// Basic sketch for trying out the Adafruit DRV8871 Breakout
#include <Arduino.h> 
#include <math.h>
#include <stdio.h>
using namespace std;


// Sensor interface:
#define AOpin  1     // Analog output - yellow
#define AOdigital 19
#define SIpin  4     // Start Integration - orange
#define CLKpin 3     // Clock - red
                     // Vcc - brown
                     // GND - black
#define LEDpin 9
#define NPIXELS 128  // No. of pixels in array
#define FASTADC 1
 // defines for setting and clearing register bits
#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#define CLK_HI sbi(PORTD, PD0)
#define CLK_LO cbi(PORTD, PD0)
#define SI_HI  sbi(PORTD, PD4)
#define SI_LO  cbi(PORTD, PD4)
uint16_t Pixel[NPIXELS+5]; // Field for measu4red values <0-1023>
int expTime;
int sat_pixels;
int analogCount;
int digitalCount;
void initSensor();
void scanSensor(int int_time, bool init_scan);
void ClockPulse();


// Motor interface
#define MOTOR_IN1 5
#define MOTOR_IN2 6
#define PWM187k 1   // 187500 Hz
#define PWM94k  2   //  93750 Hz
#define PWM47k  3   //  46875 Hz
#define PWM23k  4   //  23437 Hz
#define PWM12k  5   //  11719 Hz
#define PWM6k   6   //   5859 Hz
#define PWM3k   7   //   2930 Hz
#define PWM5        OCR3A
#define PWM6        OCR4D
#define DUTY2PWM(x)  ((255*(x))/100)
void pwm613configure(int mode);
void pwmSet6(int value);

// Computation
#define directionDown 0
#define directionUp 1
#define brwonButton 0
int getDisplacement (uint32_t centroid);
int getPwm (int displacement);
void smoothDis ();
void getDirection ();
int curCen;
int preCen;
int direction;
float force;
int magicNumber = 0;
int incomingByte = 0;  

float brownDown[] = {5.3, 29.73, 32.67, 32.34, 32.33, 32.76, 32.91, 33.56, 34.27, 34.68, 35.07, 35.29, 36.28, 36.34, 36.54, 37.05, 37.29, 37.86, 38.32, 38.12, 39.24, 40.02, 43.2, 44.36, 47.77, 49.02, 47.58, 47.76, 47.16, 46.59, 45.63, 44.71, 43.68, 42.8, 41.96, 40.21, 39.94, 39.37, 39.43, 39.39, 38.09, 38.26, 39.65, 39.77, 40.45, 42.83, 44.78, 43.4, 45.43, 49.06, 50.34, 50.55, 50.71, 51.09, 51.38, 51.49, 51.96, 52.2, 52.46, 52.86, 53.48, 53.8, 54.05, 54.46, 54.93, 55.36, 55.96, 56.3, 56.98, 57.57, 58.23, 57.87, 58.55, 59.03, 59.09, 59.81, 60.02, 60.24, 61.18, 61.64, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54, 62.54};
float brownUp[] = {0, 4.6, 14.44, 21.67, 22.44, 22.48, 23.32, 22.74, 23.27, 23.19, 23.52, 24.48, 24.4, 24.79, 25.24, 25.52, 26.52, 27.29, 28.12, 28.66, 29.31, 30.89, 33.77, 35.54, 38.0, 38.92, 38.2, 37.51, 36.66, 35.38, 33.92, 33.14, 32.29, 31.36, 29.95, 28.54, 27.79, 27.19, 27.85, 27.64, 27.02, 27.49, 28.58, 30.44, 28.22, 28.98, 30.23, 34.0, 37.72, 43.76, 44.41, 44.89, 45.17, 45.36, 45.25, 45.38, 45.78, 46.32, 46.65, 46.92, 47.17, 47.69, 47.87, 48.52, 48.53, 49.26, 49.6, 49.79, 49.95, 50.32, 50.06, 50.87, 50.52, 51.09, 51.47, 52.37, 52.41, 52.97, 53.06, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75, 53.75};
float blackDown[] = {7.134, 7.134, 16.77, 27.408, 27.256, 32.302, 36.672, 44.762, 44.894, 45.308, 45.858, 46.274, 47.064, 47.22, 47.792, 48.63, 48.674, 49.572, 49.852, 50.734, 51.002, 51.766, 52.628, 53.208, 53.494, 53.532, 54.042, 54.578, 55.038, 55.468, 55.874, 56.354, 56.978, 57.656, 58.026, 58.788, 59.35, 60.046, 60.744, 61.662, 62.3, 63.09, 63.684, 64.424, 65.472, 65.804, 66.884, 67.13, 67.906, 68.78, 69.258, 69.546, 70.108, 70.648, 70.88, 71.796, 71.796, 73.348, 74.85, 76.25, 76.942, 77.854, 78.618, 79.162, 80.378, 81.294, 82.104, 82.562, 83.44, 84.362, 85.106, 85.84, 86.674, 86.99, 87.874, 89.078, 88.51, 89.986, 90.298, 91.646, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228, 92.228};
float blackUp[] = {0.02, 0.076, 0.368, 2.016, 9.97, 14.084, 26.15, 26.874, 27.224, 27.914, 28.384, 28.648, 29.422, 29.594, 29.846, 30.73, 31.292, 32.018, 32.766, 33.684, 34.562, 34.826, 35.32, 35.85, 36.434, 37.24, 37.504, 38.36, 38.818, 39.272, 39.888, 40.278, 40.888, 41.43, 42.004, 42.63, 43.362, 44.16, 44.464, 45.102, 46.022, 46.764, 47.076, 47.886, 48.44, 48.962, 49.618, 50.01, 50.44, 51.134, 51.634, 51.782, 52.076, 52.884, 53.46, 53.842, 54.662, 56.328, 57.112, 57.964, 58.568, 58.844, 59.266, 59.39, 60.214, 60.644, 60.686, 61.428, 61.606, 61.948, 62.466, 63.296, 63.6, 63.758, 65.356, 66.606, 66.268, 66.602, 67.168, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962, 66.962};
float blueDown[] = {5.78, 12.62, 20.92, 25.72, 27.56, 27.73, 32.53, 33.77, 33.68, 36.06, 36.67, 37.02, 37.24, 37.5, 37.98, 38.26, 38.25, 38.79, 38.94, 39.49, 39.65, 39.92, 40.56, 40.95, 41.02, 41.7, 42.0, 42.24, 42.92, 43.2, 43.5, 43.61, 45.81, 48.85, 51.28, 54.28, 57.63, 58.18, 61.38, 63.9, 61.87, 54.23, 52.78, 51.7, 46.9, 48.95, 49.33, 49.3, 50.04, 50.28, 50.79, 50.96, 51.02, 51.41, 51.73, 52.21, 52.4, 52.98, 52.99, 53.58, 54.05, 54.13, 54.85, 55.08, 55.62, 55.96, 56.42, 56.65, 57.04, 57.44, 57.86, 58.02, 58.79, 58.96, 59.35, 60.03, 60.22, 61.0, 61.78, 66.05, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3, 80.3};
float blueUp[] = {0.21, 2.53, 5.14, 9.46, 10.12, 12.11, 15.72, 14.69, 18.33, 22.1, 28.01, 32.73, 34.19, 34.2, 34.43, 35.05, 33.17, 30.97, 27.82, 26.53, 25.43, 22.06, 19.39, 20.07, 19.55, 18.93, 18.64, 18.44, 19.1, 19.06, 18.79, 18.98, 18.28, 18.75, 23.71, 28.64, 33.78, 37.01, 40.61, 41.7, 41.67, 42.57, 42.45, 42.7, 43.28, 43.49, 43.97, 43.8, 43.9, 44.17, 44.36, 44.68, 44.99, 45.21, 45.72, 45.95, 46.26, 46.72, 46.71, 47.22, 47.15, 47.74, 48.05, 48.19, 48.23, 48.76, 49.09, 49.14, 49.56, 49.91, 49.89, 50.07, 50.73, 51.04, 51.04, 51.8, 52.06, 52.95, 52.86, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27, 80.27};
float greenDown[] = {29.82, 29.82, 41.65, 39.77, 39.4, 41.1, 42.21, 42.12, 44.26, 44.86, 45.14, 45.54, 46.25, 46.63, 47.01, 47.66, 48.05, 48.58, 48.99, 49.55, 49.88, 50.34, 50.73, 51.26, 51.69, 51.94, 52.67, 53.01, 53.39, 53.96, 54.13, 54.89, 57.19, 59.13, 65.41, 70.34, 72.72, 75.11, 75.68, 69.38, 68.88, 67.42, 66.19, 64.94, 63.86, 63.0, 62.56, 62.57, 62.78, 63.39, 63.79, 64.04, 64.67, 65.22, 65.43, 66.04, 66.59, 67.42, 67.55, 68.29, 68.67, 69.3, 69.75, 70.28, 70.85, 71.53, 72.15, 72.69, 73.65, 74.51, 75.02, 76.11, 76.83, 77.71, 77.84, 78.41, 79.54, 80.16, 80.21, 81.03, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43, 93.43};
float greenUp[] = {9.16, 9.16, 10.37, 15.14, 16.61, 16.7, 17.34, 18.84, 22.22, 27.51, 33.86, 39.68, 45.57, 45.9, 46.51, 45.27, 42.73, 40.57, 38.44, 37.43, 36.14, 34.82, 32.8, 31.05, 29.97, 29.41, 29.8, 29.95, 29.48, 30.04, 29.99, 29.41, 30.09, 28.94, 29.65, 29.67, 37.43, 48.8, 54.58, 57.88, 58.45, 58.81, 59.22, 59.72, 60.09, 60.46, 60.93, 61.28, 61.87, 62.25, 62.59, 63.19, 63.73, 64.25, 64.62, 64.99, 65.53, 66.08, 66.6, 67.07, 67.5, 67.91, 68.31, 68.55, 68.94, 69.36, 69.75, 70.22, 70.0, 70.26, 70.98, 71.22, 71.84, 71.94, 72.23, 72.95, 73.06, 73.61, 74.06, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62, 74.62};
float greyDown[] = {9.45, 9.45, 13.18, 38.89, 44.3, 53.52, 54.08, 54.22, 55.26, 55.54, 56.39, 57.76, 57.99, 58.99, 59.66, 60.09, 61.16, 64.85, 66.49, 69.83, 72.65, 76.29, 77.67, 79.61, 80.92, 82.06, 83.41, 84.36, 85.09, 85.77, 87.47, 87.69, 88.25, 89.04, 88.37, 87.75, 86.25, 85.24, 84.31, 83.21, 81.92, 80.06, 78.46, 76.21, 73.68, 72.29, 71.28, 72.36, 73.54, 74.42, 76.59, 80.45, 87.35, 93.09, 94.11, 95.04, 95.72, 96.84, 97.4, 98.29, 99.07, 100.05, 101.54, 102.57, 103.09, 104.02, 105.18, 106.38, 107.04, 108.08, 109.13, 109.65, 110.35, 110.95, 111.99, 113.12, 113.49, 114.95, 115.76, 116.85, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48, 118.48};
float greyUp[] = {0.03, 0, 0.96, 20.72, 28.21, 36.41, 37.44, 38.87, 39.05, 40.49, 41.34, 42.12, 42.38, 43.02, 44.06, 44.85, 47.08, 50.93, 52.99, 57.66, 60.39, 63.82, 66.43, 67.51, 67.83, 69.09, 69.54, 71.06, 71.28, 72.83, 72.03, 72.68, 72.71, 71.5, 70.49, 69.08, 66.97, 66.22, 64.66, 63.74, 61.77, 58.65, 56.37, 51.85, 49.84, 50.42, 52.41, 55.32, 56.45, 61.56, 66.42, 73.17, 80.07, 84.09, 84.45, 85.01, 85.33, 86.31, 86.69, 88.0, 88.35, 89.15, 90.14, 90.94, 91.6, 91.93, 92.72, 93.44, 93.96, 94.95, 95.37, 96.3, 97.0, 97.33, 98.33, 98.87, 99.27, 100.06, 101.0, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68, 101.68};
float redDown[] = {5.5, 30.08, 28.83, 33.31, 34.02, 34.35, 35.29, 35.3, 35.38, 35.57, 35.67, 35.98, 36.51, 36.76, 36.94, 37.12, 37.49, 38.06, 37.99, 38.74, 38.62, 38.59, 38.98, 38.98, 38.59, 39.35, 39.72, 39.92, 40.67, 41.23, 41.2, 41.35, 41.66, 41.9, 42.28, 42.76, 43.36, 43.36, 43.85, 43.58, 43.9, 44.35, 44.63, 45.25, 45.87, 46.85, 47.81, 48.58, 49.25, 50.46, 52.69, 53.62, 54.29, 54.58, 54.78, 55.79, 55.78, 55.18, 55.97, 56.04, 57.16, 57.03, 59.06, 58.55, 58.9, 59.42, 59.67, 60.6, 60.36, 60.67, 61.86, 62.0, 62.03, 62.07, 61.22, 61.11, 60.79, 61.76, 62.13, 62.39, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79, 62.79};
float redUp[] = {0.12, 0, 6.65, 9.87, 15.28, 22.19, 22.39, 23.09, 22.98, 23.24, 24.08, 24.19, 25.03, 25.07, 25.64, 25.81, 26.03, 26.54, 26.99, 27.39, 28.0, 28.62, 29.53, 30.11, 30.2, 30.53, 31.53, 31.52, 31.83, 32.34, 33.33, 33.89, 33.07, 34.08, 34.3, 34.67, 35.42, 35.31, 35.98, 36.62, 36.81, 37.38, 37.81, 37.85, 38.35, 38.57, 38.85, 39.49, 39.38, 39.92, 40.38, 42.03, 42.44, 42.61, 43.06, 42.58, 42.54, 43.44, 42.81, 43.51, 43.48, 43.95, 45.1, 45.04, 45.38, 44.9, 45.49, 45.7, 45.37, 45.63, 45.9, 46.53, 47.81, 47.85, 48.58, 48.38, 48.2, 48.43, 47.67, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76, 48.76};
float transDown[] = {22, 22.302, 31.376, 35.132, 36.21, 36.76, 37.436, 38.17, 38.634, 39.23, 40.102, 40.972, 42.562, 44.734, 47.06, 50.064, 51.824, 53.906, 56.186, 57.228, 58.124, 59.47, 60.322, 61.036, 62.452, 62.848, 64.176, 64.39, 64.638, 64.848, 63.826, 63.226, 62.514, 62.218, 61.132, 59.672, 59.022, 57.25, 55.564, 52.612, 51.78, 52.04, 52.028, 53.206, 53.78, 55.122, 58.868, 62.12, 64.756, 66.07, 66.73, 67.728, 68.428, 69.032, 69.786, 70.528, 71.254, 72.258, 72.802, 73.644, 74.222, 75.048, 75.61, 76.094, 76.874, 77.754, 78.49, 79.012, 79.922, 80.628, 81.062, 81.898, 82.88, 83.878, 85.254, 85.996, 86.206, 87.196, 88.104, 88.776, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798, 89.798};
float transUp[] = {0.458, 11.454, 10.644, 16.906, 19.32, 20.638, 20.992, 23.238, 24.16, 24.894, 25.952, 26.826, 28.834, 33.226, 36.068, 38.758, 41.438, 43.458, 44.414, 45.322, 46.346, 47.192, 47.838, 48.54, 48.842, 49.618, 49.724, 49.81, 49.874, 48.954, 48.666, 47.408, 46.568, 45.312, 44.656, 41.72, 38.296, 37.284, 34.488, 31.71, 31.088, 32.658, 33.702, 35.874, 40.13, 46.348, 51.524, 57.134, 61.244, 62.578, 63.388, 63.964, 64.828, 65.078, 65.778, 66.698, 67.036, 67.75, 68.412, 69.218, 69.896, 70.458, 71.14, 71.866, 72.664, 73.082, 74.018, 74.422, 74.95, 75.646, 76.35, 76.732, 77.408, 77.888, 78.13, 78.76, 79.6, 80.102, 80.908, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89, 81.89};
float whiteDown[] = {6.17, 35.92, 39.26, 39.42, 40.17, 42.18, 43.69, 43.89, 45.46, 45.69, 46.27, 46.7, 47.03, 47.36, 47.8, 48.33, 48.48, 48.98, 49.51, 50.11, 50.24, 50.79, 51.23, 51.63, 52.25, 52.69, 53.08, 52.8, 52.96, 53.51, 53.95, 54.48, 54.84, 57.12, 58.38, 59.93, 63.24, 61.57, 65.4, 68.11, 70.5, 70.96, 70.72, 69.89, 66.54, 66.05, 63.23, 61.97, 61.78, 62.06, 62.53, 63.16, 63.51, 63.66, 64.29, 64.67, 64.99, 65.56, 66.11, 66.6, 66.9, 67.24, 67.88, 68.54, 68.87, 69.5, 70.27, 70.7, 70.97, 71.71, 71.88, 72.54, 72.92, 73.48, 73.89, 74.49, 74.98, 75.72, 75.97, 76.75, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32, 77.32};
float whiteUp[] = {0.12, 0, 0.13, 0.16, 8.79, 18.48, 20.57, 23.07, 25.98, 28.63, 31.38, 35.38, 38.64, 41.14, 42.47, 45.54, 44.96, 44.4, 44.41, 42.74, 40.68, 39.89, 38.2, 36.38, 35.49, 35.36, 34.44, 34.7, 34.9, 34.71, 34.6, 35.32, 35.47, 35.23, 35.87, 36.54, 37.16, 38.36, 39.11, 40.38, 42.9, 50.19, 55.92, 57.37, 58.58, 59.35, 59.65, 60.22, 60.65, 61.01, 61.54, 61.95, 62.28, 62.81, 63.08, 63.71, 64.12, 64.47, 64.96, 65.42, 65.8, 66.47, 66.88, 67.45, 67.86, 68.68, 68.93, 69.37, 70.03, 70.21, 70.81, 71.18, 71.62, 72.14, 72.38, 72.94, 73.22, 73.91, 74.66, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94, 74.94};
float appleDown[] = {2.27, 7.725, 13.18, 17.895, 22.61, 26.235, 29.86, 32.985, 36.11, 38.254999999999995, 40.4, 42.36, 44.32, 45.989999999999995, 47.66, 48.815, 49.97, 50.739999999999995, 51.51, 52.06, 52.61, 52.54, 52.47, 52.175, 51.88, 51.34, 50.8, 49.879999999999995, 48.96, 47.985, 47.01, 45.985, 44.96, 43.545, 42.13, 40.72, 39.31, 37.63, 35.95, 34.285, 32.62, 31.13, 29.64, 28.189999999999998, 26.74, 25.494999999999997, 24.25, 26.435000000000002, 28.62, 30.549999999999997, 32.48, 34.235, 35.99, 37.875, 39.76, 41.879999999999995, 44.0, 45.84, 47.68, 49.415, 51.15, 52.599999999999994, 54.05, 55.385, 56.72, 57.61, 58.5, 59.61, 60.72, 61.989999999999995, 63.26, 64.53, 65.8, 67.3, 68.8, 76.555, 84.31, 96.38499999999999, 108.46, 126.93, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4, 145.4};
float appleUp[] = {0, 4.460000000000001, 8.96, 13.265, 17.57, 21.275, 24.98, 27.995, 31.01, 33.185, 35.36, 37.18, 39.0, 40.66, 42.32, 43.58, 44.84, 45.695, 46.55, 46.97, 47.39, 47.39, 47.39, 47.09, 46.79, 46.17, 45.55, 44.79, 44.03, 43.120000000000005, 42.21, 41.17, 40.13, 38.955, 37.78, 36.475, 35.17, 33.730000000000004, 32.29, 30.79, 29.29, 27.77, 26.25, 24.939999999999998, 23.63, 22.490000000000002, 21.35, 21.810000000000002, 22.27, 24.259999999999998, 26.25, 28.275, 30.3, 32.32, 34.34, 36.385000000000005, 38.43, 40.239999999999995, 42.05, 43.76, 45.47, 46.82, 48.17, 49.36, 50.55, 51.64, 52.73, 53.739999999999995, 54.75, 56.0, 57.25, 58.78, 60.31, 62.150000000000006, 63.99, 67.87, 71.75, 85.81, 99.87, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635, 122.635};
float samsungDown[] = {2.35, 12.205, 22.06, 26.72, 31.38, 35.589999999999996, 39.8, 43.285, 46.77, 49.510000000000005, 52.25, 54.435, 56.62, 58.28, 59.94, 61.2, 62.46, 63.334999999999994, 64.21, 64.8, 65.39, 65.64, 65.89, 65.775, 65.66, 65.17, 64.68, 63.915000000000006, 63.15, 62.11, 61.07, 59.955, 58.84, 57.535, 56.23, 54.81, 53.39, 51.855000000000004, 50.32, 48.685, 47.05, 45.275, 43.5, 43.72, 43.94, 44.31, 44.68, 45.254999999999995, 45.83, 46.86, 47.89, 48.72, 49.55, 50.254999999999995, 50.96, 51.57, 52.18, 52.765, 53.35, 54.105000000000004, 54.86, 55.825, 56.79, 58.120000000000005, 59.45, 61.115, 62.78, 64.75, 66.72, 68.88499999999999, 71.05, 73.495, 75.94, 79.815, 83.69, 92.94, 102.19, 115.3, 128.41, 142.445, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48, 156.48};
float samsungUp[] = {7.5, 13.575, 19.65, 23.82, 27.99, 31.775, 35.56, 38.730000000000004, 41.9, 44.644999999999996, 47.39, 49.59, 51.79, 53.445, 55.1, 56.225, 57.35, 58.19, 59.03, 59.595, 60.16, 60.459999999999994, 60.76, 60.65, 60.54, 60.045, 59.55, 58.775, 58.0, 56.95, 55.9, 54.739999999999995, 53.58, 52.269999999999996, 50.96, 49.55, 48.14, 46.615, 45.09, 43.355000000000004, 41.62, 39.815, 38.01, 37.905, 37.8, 38.035, 38.27, 38.730000000000004, 39.19, 39.75, 40.31, 40.935, 41.56, 42.150000000000006, 42.74, 43.290000000000006, 43.84, 44.540000000000006, 45.24, 46.115, 46.99, 48.21, 49.43, 51.03, 52.63, 54.620000000000005, 56.61, 58.93, 61.25, 63.865, 66.48, 69.33000000000001, 72.18, 76.37, 80.56, 90.355, 100.15, 113.41, 126.67, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575, 141.575};
float mechwhiteDown[] = {1.34, 4.6, 7.42, 13.08, 18.42, 23.25, 27.19, 31.87, 35.95, 40.09, 43.83, 47.19, 50.39, 53.57, 55.96, 58.72, 60.73, 62.89, 64.34, 65.57, 66.82, 67.79, 68.13, 68.95, 69.1, 69.39, 69.43, 69.13, 69.57, 69.49, 69.77, 69.68, 69.58, 69.89, 69.69, 69.85, 69.62, 69.23, 69.14, 68.22, 67.17, 66.43, 65.59, 64.65, 64.69, 63.73, 63.51, 62.32, 61.58, 60.96, 59.79, 59.68, 59.09, 58.93, 58.03, 57.58, 56.34, 55.46, 55.22, 54.47, 54.42, 53.47, 53.36, 52.72, 52.88, 52.41, 51.99, 52.39, 52.42, 52.39, 52.65, 52.61, 52.77, 53.08, 53.42, 54.17, 54.96, 55.83, 57.02, 64.99, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05, 78.05};
float mechwhiteUp[] = {0.07, 1.27, 3.04, 6.77, 10.49, 14.59, 18.24, 21.64, 25.25, 28.33, 31.55, 34.82, 37.31, 39.71, 41.71, 43.92, 45.41, 47.04, 48.24, 49.3, 50.13, 50.74, 51.18, 51.77, 51.63, 51.75, 51.92, 51.79, 52.02, 51.7, 51.67, 51.74, 51.46, 51.8, 51.62, 51.34, 51.26, 50.78, 50.27, 49.87, 49.29, 49.07, 48.5, 47.88, 47.72, 46.71, 46.75, 45.64, 44.95, 44.37, 43.53, 43.17, 42.45, 41.97, 41.55, 41.02, 40.87, 40.14, 39.5, 39.33, 38.7, 38.21, 37.78, 37.83, 37.58, 38.04, 38.02, 38.09, 38.25, 38.25, 38.54, 39.13, 39.6, 40.41, 41.55, 42.45, 43.23, 43.74, 46.52, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96, 57.96};
#define lightButton 1
#define heavyButton 2
#define lockButton 3
#define tactileButton 4
#define peakButton 5
#define multiClickButton 6
#define stuckButton 7
#define weirdButton 8
#define diffPeakButton 9
#define multiStateButton 10
int buttonState = 0;
int pressDown = 0;
#define weirdButton 8

// set Button Type
int buttonType = lightButton;

void setup() {
    // motor setup 
    direction = directionDown;
    pinMode(MOTOR_IN1, OUTPUT);
    pinMode(MOTOR_IN2,OUTPUT);
    // Fast PWM mode
    pwm613configure(PWM187k);
    pwmSet6(0);
    analogWrite(MOTOR_IN1, 0);
    analogWrite(MOTOR_IN2, 132);

    // sensor setup
    pinMode(SIpin, OUTPUT);
    pinMode(CLKpin, OUTPUT);
    //pinMode (AOpin, INPUT);
    pinMode(AOdigital, INPUT);
    pinMode(LEDpin, OUTPUT);
    pinMode(7, OUTPUT);
    digitalWrite(7, LOW);
    digitalWrite(LEDpin, HIGH);
    initSensor();

#if FASTADC
  // set prescale to 16
    //sbi(ADCSRA,ADPS2);
    //cbi(ADCSRA,ADPS1);
    //cbi(ADCSRA,ADPS0);
    ADCSRA = (ADCSRA & B11111000) | 4; 
#endif

    Serial.begin (115200);
    Serial.print("serial start");
    expTime = 100;
    sat_pixels = 0;
    bool right_exposure = false;
    while (!right_exposure)
    {
        scanSensor(expTime, true);
        if(sat_pixels < 110)
            expTime += 5;
        else
            right_exposure = true;
    }
    curCen = 0;
    preCen = 0;
    direction = 0;
}
 
void loop() {
    
    long i;
    analogCount = 0;
    digitalCount = 0;
    uint32_t centroid = 0;
    uint32_t sumVal = 0;
    int displacement = 0;
    int pwm = 0;
    digitalWrite(7, HIGH);
    scanSensor(expTime, false);
    digitalWrite(7, LOW);
    if (Serial.available() > 0) 
    {
        // read the incoming byte:
        incomingByte = Serial.read();

        // say what you got:
        Serial.print("I received: ");
        Serial.println(incomingByte, DEC);
        switch (incomingByte)
        {
            case 0:
                Serial.print("set light button");
                buttonType = lightButton;
                break;
            case 1:
                Serial.print("set heavy button");
                buttonType = heavyButton;
                break;
            case 2:
                Serial.print("set lock button");
                buttonType = lockButton;
                break;
            case 3:
                Serial.print("set tactile button");
                buttonType = tactileButton;
                break;
            case 4:
                Serial.print("set peak button");
                buttonType = peakButton;
                break;
            case 5:
                Serial.print("set multi Click button");
                buttonType = multiClickButton;
                break;
            case 6:
                Serial.print("set stuck button");
                buttonType = stuckButton;
                break;
            case 7:
                Serial.print("set weird button");
                buttonType = weirdButton;
                break;
            case 8:
                Serial.print("set 2peak button");
                buttonType = diffPeakButton;
                break;
            case 9:
                Serial.print("set weird button");
                buttonType = multiStateButton;
                break;
        }
    }
    for (i = 0; i < NPIXELS; i++) 
    {
        int val = 1023 - Pixel[i];
        centroid += i* val;
        sumVal += val;        
    }
    if (sumVal > 0)
        centroid = centroid*100 / sumVal;

    displacement = getDisplacement(centroid);
    curCen = displacement;
    //smoothDis();
    getDirection();
    if (buttonType == multiStateButton)
    {   
        if (displacement >3900 && pressDown ==0)
        {
            Serial.println("CLICK!");
            pressDown = 1;
        }

        if (direction==directionUp && displacement<3700 && pressDown==1)
        {
            Serial.println("CHANGE STATE");
            if (buttonState ==0)
            {buttonState = 1;Serial.println("CHANGE STATE TO 1");}
            else if (buttonState ==1)
            {buttonState = 0;}
            Serial.print("STATE: ");
            Serial.println(buttonState);
            pressDown=0;
        }
    }
    pwm = getPwm(curCen);
    analogWrite(MOTOR_IN2, pwm);
    /*
    Serial.print(centroid);
    Serial.print("\t");
    Serial.print(displacement);
    Serial.print("\t");
    Serial.print(direction);
    Serial.print("\t");
    Serial.print(force);
    Serial.print("\t");
    Serial.println(pwm);*/
    preCen = curCen;
    
}

void initSensor()
{
    // Initialize two Arduino pins as digital output:
    pinMode(CLKpin, OUTPUT);
    pinMode(SIpin, OUTPUT);

    // Clock out any existing SI pulse through the ccd register:
    for(int i=0;i< NPIXELS+4;i++)
    {
        ClockPulse();
    }

    // Create a new SI pulse and clock out that same SI pulse through the sensor register:
    SI_HI;
    delayMicroseconds(1);
    CLK_HI;
    delayMicroseconds(1);
    SI_LO;
    delayMicroseconds(1);
    CLK_LO;

    for(int i=0;i< NPIXELS+4;i++)
    {
        ClockPulse();
    }
}

// This function generates an outgoing clock pulse from the Arduino digital pin 'CLKpin'. This clock
// pulse is fed into pin 3 of the linear sensor:
void ClockPulse()
{
    delayMicroseconds(1);
    CLK_HI;
    CLK_LO;
}

void scanSensor(int int_time, bool init_scan) // maximum 16383  = 16.3 ms
{
    // Stop the ongoing integration of light quanta from each photodiode by clocking in a
    // SI pulse:
    // Create a new SI pulse and clock out that same SI pulse through the sensor register:
    SI_HI;
    delayMicroseconds(1);
    CLK_HI;
    delayMicroseconds(1);
    SI_LO;
    delayMicroseconds(1);
    CLK_LO;

    // Next, a new measuring cycle is starting once 18 clock pulses have passed. At
    // that time, the photodiodes are once again active. We clock out the SI pulse through
    // the NPIXELS bit register in order to be ready to halt the ongoing measurement at our will
    // (by clocking in a new SI pulse):
    for(int i = 0; i < NPIXELS+4; i++)
    {
        if(i==18)
        {
            // Now the photodiodes goes active..
            // An external trigger can be placed here
        }
        ClockPulse();
    }

    // set integration time
    delayMicroseconds(int_time); // Stop the ongoing integration of light quanta from each photodiode by clocking in a SI pulse
    
    // into the sensors register:
    // Create a new SI pulse and clock out that same SI pulse through the sensor register:
    SI_HI;
    delayMicroseconds(1);
    CLK_HI;
    delayMicroseconds(1);
    SI_LO;
    delayMicroseconds(1);
    CLK_LO;
    
    // Next, read all 256 pixels in parallell. Store the result in the array. Each clock pulse
    // causes a new pixel to expose its value on the two outputs:
    if (init_scan)
    {
        sat_pixels = 0;
    }
    digitalWrite(7, HIGH);
    
    for(int i=0; i < NPIXELS; i++)
    {
        delayMicroseconds(1);// <-- We add a delay to stabilize the AO output from the sensor
        if (i%4==0) 
        {
            if (digitalRead(AOdigital) == HIGH)
            {
                Pixel[i] = 1023;
                digitalCount++;
            }
            else
            {
                Pixel[i] = 0;
                //Pixel[i] = analogRead(AOpin);
                analogCount ++;
            }
            //Pixel[i] = digitalRead(AOdigital);
            Pixel[i+1] = Pixel[i];
            Pixel[i+2] = Pixel[i]; 
            Pixel[i+3] = Pixel[i];  
        }
         
        ClockPulse();
        if (init_scan && Pixel[i]>=1023)
            sat_pixels += 1;
    }
    digitalWrite(7, LOW);
}

void pwm613configure(int mode)
{
    // TCCR4A configuration
    TCCR4A=0;
    // TCCR4B configuration
    TCCR4B=mode;
    // TCCR4C configuration
    TCCR4C=0;
    // TCCR4D configuration
    TCCR4D=0;
    // TCCR4D configuration
    TCCR4D=0;
    // PLL Configuration
    // Use 96MHz / 2 = 48MHz
    PLLFRQ=(PLLFRQ&0xCF)|0x30;
    // PLLFRQ=(PLLFRQ&0xCF)|0x10; // Will double all frequencies
    // Terminal count for Timer 4 PWM
    OCR4C=255;
}

void pwmSet6(int value)
{
    OCR4D=value;   // Set PWM value
    DDRD|=1<<7;    // Set Output Mode D7
    TCCR4C|=0x09;  // Activate channel D
}

int getDisplacement (uint32_t centroid)
{
    float intercept = -49.926437861629;
    float trans = 0.0577493694673854;
    float displacement = intercept + trans * (float) centroid;
    displacement *= 10;
    int intDis = 0;
    if (displacement <100)
        displacement = 100;
    if (displacement > 4000)
        displacement = 4000;
    
    if (displacement - round(displacement) >= 0.5) 
        intDis = (int)round(displacement) +1;
    else 
        intDis = (int)round(displacement);
    
    //if ((intDis %50) < 25 )
    //    intDis = intDis - (intDis %50);
    //else
    //    intDis = intDis - (intDis %50) +50;
    return intDis; // return in 10mm
}

int getPwm (int displacement)
{
    float pwm;
    int getArray = displacement/50;
    int adjustment = displacement%50;
    switch (buttonType)
    {
        case lightButton:
            pwm = 140+displacement/400;
            break;
        case heavyButton:
            pwm = 160+displacement/400;
            break;
        case lockButton:
            if (displacement < 2000)
            {pwm = 165;}
            else
            {pwm = 140;}
            break;
        case tactileButton:
            if (displacement < 3000)
            {pwm = 140 + displacement/150;}
            else
            {pwm = 140;}
            break;
        case peakButton:
            if (displacement < 2000)
            {pwm = 140 + displacement/100;}
            else
            {pwm = 160-(displacement-2000)/100;}
            break;
        case multiClickButton:
            if (displacement < 1000)
            {pwm = 140+displacement/200;}
            else if (displacement > 1000 && displacement < 2000)
            {pwm = 140;}
            if (displacement > 2000 && displacement < 3000)
            {pwm = 140+(displacement-2000)/100;}
            else if (displacement > 3000)
            {pwm = 140;}
            break;
        case multiStateButton:
            if (buttonState == 0)
            {pwm = 165;}
            if (buttonState == 1)
            {
                if (displacement >3700)
                {pwm = 140;}
                else
                {pwm = 137;}
            }
            break;
        case weirdButton:
            if ((direction == directionDown && displacement<3900) || displacement<300)
            {pwm = 140;}
            else
            {pwm = 175;}
            break;
        case stuckButton:
            if ( (direction == directionDown && displacement<4000) || displacement<200)
            {pwm = 175;}
            else
            {pwm = 138;}
            break;
        case diffPeakButton:
            if (direction == directionDown)
                if (displacement < 3500)
                {pwm = 140 + displacement/175;}
                else
                {pwm = 140;}
            else
                if (displacement < 800)
                {pwm = 145 + displacement/40;}
                else
                {pwm = 145;}
            break;
        default:
            if (direction == directionDown)
            {
                force = mechwhiteDown[getArray] + ((mechwhiteDown[getArray+1]-mechwhiteDown[getArray])/50)*adjustment;  
                pwm = 136.946561518423 + 0.186690803264018 * force;
            }
            else 
            {
                force = mechwhiteUp[getArray] + ((mechwhiteUp[getArray+1]-mechwhiteUp[getArray])/50)*adjustment;
                pwm = 137.786871765628 + 0.189973389165936 * force;
            }
            //pwm = 150;
    }
    if (pwm - round(pwm) >=0.5)
        return round(pwm) + 1;
    else
        return round(pwm);
}

void smoothDis()
{
    if (abs(curCen - preCen)<7)
        curCen = preCen;
}

void getDirection()
{
    if (curCen - preCen >2)  
        direction = directionDown;
    else if (curCen - preCen < -2)
        direction = directionUp;
}
