package services

import javax.inject.{Inject, Singleton}

import db.Sentences.sentences
import db.TestSections.testSections
import db.Participants.participants
import db.Keystrokes.keystrokes
import db.Logdatas.logdatas
import models.{Keystroke, Participant, Sentence, TestSection, Logdata}
import play.api.db.slick.DatabaseConfigProvider

import scala.concurrent.{Future, Promise}
import slick.driver.JdbcProfile
import slick.driver.MySQLDriver.api._

import scala.concurrent.ExecutionContext.Implicits.global

trait DatabaseService {
  def getRandomSentence(id: Int): Future[Seq[Sentence]]

  def updateDevice(testSectionId: Int, device: String): Future[Int]
  def updateUserInput(testSectionId: Int, userInput: String): Future[Int]
  def updateInputTime(testSectionId: Int, time: Long, wpm: Double, inputLength: Int, potentialWpm: Double, potentialLen: Int): Future[Int]
  def updateErrorRate(testSectionId: Int, editDistance: Int, errorRate: Double, errorLen: Int): Future[Int]
  def updateBrowserData(participantId: Int, ipAddress: String, 
          browserString: Option[String], browserLanguage: Option[String],
          device: Option[String],
          screen_w: Option[Int], screen_h: Option[Int]): Future[Int]
  def updatePersonalData(participantId: Int, 
          age: Int, gender: String, nativeLanguage: String, 
          hasTakenTypingCourse: Boolean, 
          timeSpentTyping: String, 
          typeEnglish: String, 
          fingers: String, layout: String, keyboardType: String, 
          usingApp: String, 
          usingFeatures: String): Future[Int]

  def updateUserStats(participantId: Int, wpm: Double, errorRate: Double)

  def getTestSections(participantId: Int): Future[Seq[TestSection]]
  def getKeystrokes(testSectionId: Int): Future[Seq[Keystroke]]
  def getFirstKeystroke(testSectionId: Int): Future[Seq[Keystroke]]
  def getDataForError(testSectionId: Int): Future[Seq[(Option[String], String)]]
  def getSentence(testSectionId: Int): Future[Seq[Sentence]]

  def getBiggestError(participantId: Int): Future[Seq[(Option[String], String, Option[Int], Option[Double])]]
  def getFastestSentence(participantId: Int): Future[Seq[(Option[String], Option[Int], Option[Double])]]
  def getSlowestSentence(participantId: Int): Future[Seq[(Option[String], Option[Int], Option[Double])]]

  def addParticipant(p: Participant): Future[Int]
  def addTestSection(ts: TestSection): Future[Int]
  def addKeystroke(ks: Keystroke)
  def addKeystrokes(ks: Seq[Keystroke]): Future[Option[Int]]
  def addLogdata(ld: Seq[Logdata]): Future[Option[Int]]  
  //def addOrientation(ld: Seq[Orientation]): Future[Option[Int]]
  def listTestSections(device: String): Future[Seq[TestSection]]
  def getWpmHistogramData(device: String): Future[Seq[(Int, Int)]]
  def getErrorRateHistogramData(device: String): Future[Seq[(Int, Int)]]
}

@Singleton
class DatabaseServiceImpl @Inject()(dbConfigProvider: DatabaseConfigProvider) extends DatabaseService {
  val dbConfig = dbConfigProvider.get[JdbcProfile]
  val rand = SimpleFunction.nullary[Double]("rand")

  def getRandomSentence(id: Int): Future[Seq[Sentence]] = {
    val participantSentenceIds = for {
        (t, s) <- testSections join sentences on (_.sentenceId === _.id) if t.participantId === id
      } yield s.id
    val unusedSentences = sentences.filterNot(_.id in participantSentenceIds)

    val rand = SimpleFunction.nullary[Double]("rand")
    dbConfig.db.run(unusedSentences.sortBy(x => rand).take(1).result)
  }

  def updateDevice(testSectionId: Int, device: String): Future[Int] = {
    val q = for { ts <- testSections if ts.id === testSectionId} yield ts.device
    dbConfig.db.run(q.update(Some(device)))
  }

  def updateUserInput(testSectionId: Int, userInput: String): Future[Int] = {
    val q = for { ts <- testSections if ts.id === testSectionId} yield ts.userInput
    dbConfig.db.run(q.update(Some(userInput)))
  }

  def updateInputTime(testSectionId: Int, time: Long, wpm: Double, inputLength: Int, potentialWpm: Double, potentialLen: Int): Future[Int] = {
    val q = for { ts <- testSections if ts.id === testSectionId} yield (ts.inputTime, ts.wpm, ts.inputLength, ts.potentialWpm, ts.potentialLength)
    dbConfig.db.run(q.update(Some(time), Some(wpm), Some(inputLength), Some(potentialWpm), Some(potentialLen)))
  }

  def updateErrorRate(testSectionId: Int, editDistance: Int, errorRate: Double, errorLen: Int): Future[Int] = {
    val q = for { ts <- testSections if ts.id === testSectionId} yield (ts.errorRate, ts.editDistance, ts.errorLen)
    dbConfig.db.run(q.update(Some(errorRate), Some(editDistance), Some(errorLen)))
  }

  def updateBrowserData(participantId: Int, ipAddress: String, 
    browserString: Option[String], browserLanguage: Option[String],
    device: Option[String],
    screen_w: Option[Int], screen_h: Option[Int]): Future[Int] = {
    val q = for { p <- participants if p.id === participantId} yield (p.ipAddress, p.browserString, p.browserLanguage, p.device, p.screen_w, p.screen_h)
    dbConfig.db.run(q.update(Some(ipAddress), browserString, browserLanguage, device, screen_w, screen_h))
  }

  def updatePersonalData(participantId: Int, 
      age: Int, gender: String, nativeLanguage: String, 
      hasTakenTypingCourse: Boolean, 
      timeSpentTyping: String, 
      typeEnglish: String, 
      fingers: String, layout: String, keyboardType: String, 
      usingApp: String, 
      usingFeatures: String): Future[Int] = {
    val q = for { p <- participants if p.id === participantId} yield (
        p.age, p.gender, p.nativeLanguage, 
        p.hasTakenTypingCourse, 
        p.timeSpentTyping, 
        p.typeEnglish,
        p.fingers, p.layout, p.keyboardType, 
        p.usingApp, 
        p.usingFeatures)
    dbConfig.db.run(q.update(
        Some(age), Some(gender), Some(nativeLanguage),
        Some(hasTakenTypingCourse), 
        Some(timeSpentTyping), 
        Some(typeEnglish), 
        Some(fingers), Some(layout), Some(keyboardType),  
        Some(usingApp), 
        Some(usingFeatures)))
  }

  def updateUserStats(participantId: Int, wpm: Double, errorRate: Double) = {
    val q = for { p <- participants if p.id === participantId } yield (p.wpm, p.errorRate)
    dbConfig.db.run(q.update(Some(wpm), Some(errorRate)))
  }

  def getTestSections(participantId: Int): Future[Seq[TestSection]] = {
    dbConfig.db.run(testSections.filter(_.participantId === participantId).filter(_.userInput.isDefined).result)
  }

  def getKeystrokes(testSectionId: Int): Future[Seq[Keystroke]] = {
    dbConfig.db.run(keystrokes.filter(_.testSectionId === testSectionId).result)
  }

  def getFirstKeystroke(testSectionId: Int): Future[Seq[Keystroke]] = {
    dbConfig.db.run(keystrokes.filter(_.testSectionId === testSectionId).sortBy(_.pressTime).take(1).result)
  }

  def getSentence(testSectionId: Int): Future[Seq[Sentence]] = {
    val q = for {
      (ts, s) <- testSections join sentences on (_.sentenceId === _.id) if ts.id === testSectionId
    } yield s
    dbConfig.db.run(q.result)
  }

  def getBiggestError(participantId: Int): Future[Seq[(Option[String], String, Option[Int], Option[Double])]] = {
    val q = for {
      max_ts <- testSections.filter(_.participantId === participantId).sortBy(_.errorRate.desc).take(1)
      s <- sentences.filter(_.id === max_ts.sentenceId)
    } yield (max_ts.userInput, s.sentence, max_ts.editDistance, max_ts.wpm)
    dbConfig.db.run(q.result)
  }

  def getFastestSentence(participantId: Int): Future[Seq[(Option[String], Option[Int], Option[Double])]] = {
    val q = for {
      max_ts <-testSections.filter(_.participantId === participantId).sortBy(_.wpm.desc).take(1)
    } yield (max_ts.userInput, max_ts.editDistance, max_ts.wpm)
    dbConfig.db.run(q.result)
  }

  def getSlowestSentence(participantId: Int): Future[Seq[(Option[String], Option[Int], Option[Double])]] = {
    val q = for {
      max_ts <-testSections.filter(_.participantId === participantId).sortBy(_.wpm.asc).take(1)
    } yield (max_ts.userInput, max_ts.editDistance, max_ts.wpm)
    dbConfig.db.run(q.result)
  }

  def addParticipant(p: Participant) = {
    dbConfig.db.run(participants returning participants.map(_.id) += p)
  }

  def addTestSection(ts: TestSection) = {
    dbConfig.db.run(testSections returning testSections.map(_.id) += ts)
  }

  def addKeystroke(ks: Keystroke) = {
     dbConfig.db.run(keystrokes += ks)
  }

  def addKeystrokes(ks: Seq[Keystroke]): Future[Option[Int]] = {
    dbConfig.db.run(keystrokes ++= ks)
  }

  def getDataForError(testSectionId: Int): Future[Seq[(Option[String], String)]] = {
    val q = for {
      (ts, s) <- testSections join sentences on(_.sentenceId === _.id) if ts.id === testSectionId
    } yield (ts.userInput, s.sentence)
    dbConfig.db.run(q.result)
  }

  def addLogdata(ld: Seq[Logdata]) = {
    dbConfig.db.run(logdatas ++= ld)
  }

  /**
   * List all the test sections 
   */
  def listTestSections(device: String): Future[Seq[TestSection]] = {
    dbConfig.db.run(testSections.filter(_.device === device).filter(_.userInput.isDefined).result)
  }

  def getWpmHistogramData(device: String): Future[Seq[(Int, Int)]] = {
    val WPM_MAX_MOBILE: Int = 100
    val WPM_MAX_DESKTOP: Int = 150
    val WPM_MIN: Int = 0 // included
    val WPM_MAX: Int = if (device == "mobile") WPM_MAX_MOBILE else WPM_MAX_DESKTOP // excluded
    val WPM_STEP_SIZE: Int = 5
    dbConfig.db.run(sql"""SELECT FLOOR(WPM/$WPM_STEP_SIZE)*$WPM_STEP_SIZE AS bucket, COUNT(*) AS count FROM PARTICIPANTS WHERE WPM IS NOT NULL AND $WPM_MIN <= WPM AND WPM < $WPM_MAX AND DEVICE = $device GROUP BY bucket""".as[(Int, Int)])
  }

  def getErrorRateHistogramData(device: String): Future[Seq[(Int, Int)]] = {
    val ERROR_RATE_MAX_MOBILE: Int = 20
    val ERROR_RATE_MAX_DESKTOP: Int = 20
    val ERROR_RATE_MIN: Int = 0 // included
    val ERROR_RATE_MAX: Int = if (device == "mobile") ERROR_RATE_MAX_MOBILE else ERROR_RATE_MAX_DESKTOP // excluded
    val ERROR_RATE_STEP_SIZE: Int = 1
    dbConfig.db.run(sql"""SELECT FLOOR(ERROR_RATE/$ERROR_RATE_STEP_SIZE)*$ERROR_RATE_STEP_SIZE AS bucket, COUNT(*) AS count FROM PARTICIPANTS WHERE ERROR_RATE IS NOT NULL AND $ERROR_RATE_MIN <= ERROR_RATE AND ERROR_RATE < $ERROR_RATE_MAX AND DEVICE = $device GROUP BY bucket""".as[(Int, Int)])
  }
}
