package controllers

import javax.inject._

import models.TestResults
import models.TestSection
import play.api._
import play.api.data.Form
import play.api.data.Forms._
import play.api.mvc._
import play.api.routing._
import play.api.libs.json._
import play.api.libs.functional.syntax._
import play.api.libs.ws.WSClient
import services.TypingTestService

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
//import play.api.i18n.Messages.Implicits._
//import play.api.Play.current

case class LogdataSubmit(etype: String, key: String, code: String, data: String,
                        input: String, pressed: String, timestamp: Long, 
                        screen_orientation: String, 
                        device_orientation: String)
case class KeystrokeSubmit(pressTime: Long, releaseTime: Long, keycode: Int, letter: String)
case class QuestionnaireData(age: Int, gender: String, nativeLanguage: String, 
  hasTakenTypingCourse: String, timeSpentTyping: String, typeEnglish: String, 
  fingers: String, layout: String, keyboardType: String,  
  usingApp: String, 
  usingFeatures: String)
case class WpmHistogramData(bucket: Int, count: Int)
case class ErrorRateHistogramData(bucket: Int, count: Int)

@Singleton
class TypingTest @Inject()(testService: TypingTestService, ws: WSClient) extends Controller {

  //case class BrowserDataSubmit(browserString: String)

  case class QuestionnaireDataSubmit(age: Int, gender: String, nativeLanguage: String, 
    hasTakenTypingCourse: String, 
    timeSpentTyping: String, 
    typeEnglish: String, 
    fingers: String, layout: String, keyboardType: String, 
    usingApp: String, 
    usingFeatures: String)

  val uidMissingResponse = BadRequest(Json.toJson(Map("status" -> "Bad Request", "error" -> "uid_missing_err")))
  val tsidMissingResponse = BadRequest(Json.toJson(Map("status" -> "Bad Request", "error" -> "tsid_missing_err")))
  val validationErrorResponse = BadRequest(Json.toJson(Map("status" -> "Bad Request", "error" -> "validation_err")))

  def consent = Action { implicit request =>
    Ok(views.html.consent(request)).withHeaders("P3P" -> "CP=\"Deprecated\"")
  }

  def instructions = Action { implicit request =>
    Ok(views.html.instructions(request)).withHeaders("P3P" -> "CP=\"Deprecated\"")
  }

  def typingTest = Action { implicit request =>
    Ok(views.html.typingtest(request)).withHeaders("P3P" -> "CP=\"Deprecated\"")
  }

  def questionnaire = Action { implicit request =>
    Ok(views.html.questionnaire(questionnaireForm)).withHeaders("P3P" -> "CP=\"Deprecated\"")
  }

  def uidErr = Action { implicit request =>
    Ok(views.html.uidErr(request)).withHeaders("P3P" -> "CP=\"Deprecated\"")
  }

  val questionnaireForm = Form(
    mapping(
      "age" -> number(min = 0, max = 120),
      "gender" -> text,
      "nativeLanguage" -> text,
      "hasTakenTypingCourse" -> text,
      "timeSpentTyping" -> text,
      "typeEnglish" -> text,
      "fingers" -> text,
      "layout" -> text,
      "keyboardType" -> text,  
      "usingApp" -> text, 
      "usingFeatures" -> text
    )(QuestionnaireData.apply)(QuestionnaireData.unapply)
  )

  def results = Action.async { implicit request =>
    request.session.get("uid") match {
      case Some(uid) => {
        for {
          fastest <- testService.getFastestSentence(uid)
          slowest <- testService.getSlowestSentence(uid)
          worst <- testService.getBiggestError(uid)
          wpm <- testService.getWPM(uid)
          errorRate <- testService.getErrorRate(uid)
        } yield {
          (fastest, slowest) match {
            case ((Some(sFastest), Some(sFastestError), Some(sFastestWpm)), (Some(sSlowest), Some(sSlowestError), Some(sSlowestWpm))) => {
              Ok(views.html.results(TestResults(wpm.toInt, errorRate, worst._1, worst._2, worst._3, worst._4, sFastest, sFastestError, sFastestWpm, sSlowest, sSlowestError, sSlowestWpm)))
                .withHeaders(CACHE_CONTROL -> "no-cache", "P3P" -> "CP=\"Deprecated\"")
            }
          }
        }
      }
      case None => Future(uidMissingResponse)
    }
  }

  def typingtestRoutes = Action { implicit request =>
    Ok(
        JavaScriptReverseRouter("typingtestRoutes")(
          routes.javascript.TypingTest.newUid,
          routes.javascript.TypingTest.sentence,
          routes.javascript.TypingTest.getWPM,
          routes.javascript.TypingTest.getErrorRate,
          routes.javascript.TypingTest.updateStats
        )
    ).as("text/javascript")
  }

  def newUid = Action.async { implicit request =>
    testService.newUid().map(s => {
      Ok(Json.toJson(Map("uid" -> s)))
        .withSession("uid" -> s)
        .withHeaders(CACHE_CONTROL -> "no-cache")
    })
  }

  def sentence(first: Boolean) = Action.async {
    implicit request => {
      (request.session.get("uid"), request.session.get("count"), request.session.get("tsid"),first) match {
        // new sentence in the middle of the test
        case (Some(uid), Some(count), _, false) => testService.newSentence(uid).map(map => {
          map.get("id") match {
            case Some(id) => Ok(Json.toJson(map + ("count" -> (count.toInt + 1).toString)))
              .withSession(request.session + ("count" -> (count.toInt + 1).toString) + ("tsid" -> id))
              .withHeaders(CACHE_CONTROL -> "no-cache")
            case _ => tsidMissingResponse
          }
        })
        // new sentence at the beginning of the test
        case (Some(uid), None, _, true) => testService.newSentence(uid).map(map => {
          map.get("id") match {
            case Some(id) => Ok(Json.toJson(map + ("count" -> (1).toString)))
              .withSession(request.session + ("count" -> "1") + ("tsid" -> id))
              .withHeaders(CACHE_CONTROL -> "no-cache")
            case _ => tsidMissingResponse
          }
        })
        // old sentence
        case (Some(uid), Some(count), Some(tsid), true) => testService.getSentence(tsid).map(map => {
          Ok(Json.toJson(map + ("count" -> count)))
            .withSession(request.session + ("count" -> count) + ("tsid" -> tsid))
            .withHeaders(CACHE_CONTROL -> "no-cache")
        })
        case _ => Future(uidMissingResponse)
      }
    }
  }

  def getWPM = Action.async {
    implicit request => {
      request.session.get("uid") match {
        case Some(uid) => testService.getWPM(uid).map(wpm => Ok(Json.toJson(Map("wpm" -> wpm.toInt))).withHeaders(CACHE_CONTROL -> "no-cache"))
        case None => Future(uidMissingResponse)
      }
    }
  }

  def updateStats = Action {
    implicit request => {
      request.session.get("uid") match {
        case Some(uid) => {
          request.session.get("questionnaire") match {
            case Some(q) => {
              testService.updateUserStats(uid)
              Ok(Json.toJson(Map("status" -> "OK", "questionnaire" -> "true"))).withSession(request.session - "count")
            }
            case None => {
              testService.updateUserStats(uid)
              Ok(Json.toJson(Map("status" -> "OK", "questionnaire" -> "false"))).withSession(request.session - "count")
            }
          }
        }
        case None => uidMissingResponse
      }
    }
  }

  def getErrorRate = Action.async {
    implicit request => {
      request.session.get("uid") match {
        case Some(uid) => testService.getErrorRate(uid).map(errorRate => Ok(Json.toJson(Map("errorRate" -> errorRate))).withHeaders(CACHE_CONTROL -> "no-cache"))
        case None => Future(uidMissingResponse)
      }
    }
  }

/*
  def saveKeystroke = Action(BodyParsers.parse.json) { request =>
    val keystroke = request.body.validate[KeystrokeSubmit]
    keystroke.fold(
      errors => validationErrorResponse,
      keystroke => {
        request.session.get("tsid") match {
          case Some(tsid) => {
            testService.submitKeystroke(keystroke.pressTime, keystroke.releaseTime, keystroke.keycode, keystroke.letter, tsid)
            Ok(Json.obj("status" -> "OK"))
          }
          case None => tsidMissingResponse
        }

      }
    )
  }
*/

  def saveKeystrokes = Action.async(BodyParsers.parse.json) {
    implicit request => {
      val keystrokes = (request.body \ "keystrokes").as[Seq[KeystrokeSubmit]]
      request.session.get("tsid") match {
        case Some(tsid) => {
          testService.submitKeystrokes(tsid, keystrokes).map {
            unit => {
              Ok(Json.obj("status" -> "OK"))
            }
          }
        }
        case None => Future(tsidMissingResponse)
      }
    }
  }

  def saveLogdata = Action.async(BodyParsers.parse.json) {
    implicit request => {
      val logs = (request.body \ "logdata").as[Seq[LogdataSubmit]]
      request.session.get("tsid") match {
        case Some(tsid) => {
          testService.submitLogdata(tsid, logs).map {
            unit => {
              Ok(Json.obj("status" -> "OK"))
            }
          }
        }
        case None => Future(tsidMissingResponse)
      }
    }
  }

  def saveUserInput = Action.async(BodyParsers.parse.json) {
    request => {
      val userInput = (request.body \ "userInput").as[String]
      val device = (request.body \ "device").as[String]
      request.session.get("tsid") match {
        case Some(tsid) => {
          testService.submitUserInput(tsid, userInput, device).map(unit => {
            Ok(Json.obj("status" -> "OK"))
          })
        }
        case None => Future(tsidMissingResponse)
      }
    }
  }


  def saveBrowserData = Action.async(BodyParsers.parse.json) {
    implicit request => {
      val screen_w = (request.body \ "screen_w").as[Int]
      val screen_h = (request.body \ "screen_h").as[Int]
      val device = (request.body \ "device").as[String]
      val browserLanguage: Option[String] = request.headers.get("Accept-Language")
      val browserString: Option[String] = request.headers.get("User-Agent")
      val remoteAddress: String     = request.remoteAddress
      request.session.get("uid") match {
        case Some(uid) => {
          testService.submitBrowserData(uid, remoteAddress, browserString, browserLanguage, Some(device), Some(screen_w), Some(screen_h)).map{
              unit => {
                Ok(Json.obj("status" -> "OK"))
              }
            }
        }
        case None => Future(tsidMissingResponse)
      }
    }
  }

  def questionnairePost = Action.async(BodyParsers.parse.json) {
    implicit request => {
      val ud = (request.body).as[QuestionnaireDataSubmit]
      request.session.get("uid") match {
        case Some(uid) => {
          testService.submitUserData(uid, 
              ud.age, ud.gender, ud.nativeLanguage,
              ud.hasTakenTypingCourse.toBoolean, 
              ud.timeSpentTyping, 
              ud.typeEnglish, 
              ud.fingers, ud.layout, ud.keyboardType, 
              ud.usingApp, 
              ud.usingFeatures).map{
              unit => {
                Ok(Json.obj("status" -> "OK")).withSession(request.session + ("questionnaire" -> "done"))
              }
            }
        }
        case None => Future(tsidMissingResponse)
      }
    }
  }

  /**
   * A REST endpoint that gets all test sections as JSON.
   */
  def getTestSectionData(device: String) = Action.async { implicit request =>
    testService.getTestSections(device).map { testsections =>
      Ok(Json.toJson(testsections))
    }
  }

  def getWpmHistogramData(device: String) = Action.async { implicit request =>
    for {
      wpmHistogramData <- testService.getWpmHistogramData(device)
    } yield {
      Ok(Json.toJson(for (entry <- wpmHistogramData) yield {
        val bucket: Int = entry._1
        val count: Int = entry._2
        WpmHistogramData(bucket, count)
      }))
    }
  }

  def getErrorRateHistogramData(device: String) = Action.async { implicit request =>
    for {
      errorRateHistogramData <- testService.getErrorRateHistogramData(device)
    } yield {
      Ok(Json.toJson(for (entry <- errorRateHistogramData) yield {
        val bucket: Int = entry._1
        val count: Int = entry._2
        ErrorRateHistogramData(bucket, count)
      }))
    }
  }

  implicit val wpmHistogramDataWrites: Writes[WpmHistogramData] = Json.writes[WpmHistogramData]
  implicit val errorRateHistogramDataWrites: Writes[ErrorRateHistogramData] = Json.writes[ErrorRateHistogramData]

  implicit val TestSectionWrites: Writes[TestSection] = (
      (JsPath \ "id").write[Int] and
      (JsPath \ "sentenceId").write[Int] and
      (JsPath \ "participantId").write[Int] and
      (JsPath \ "userInput").writeNullable[String] and
      (JsPath \ "inputTime").writeNullable[Long] and
      (JsPath \ "inputLength").writeNullable[Int] and
      (JsPath \ "errorRate").writeNullable[Double] and
      (JsPath \ "editDistance").writeNullable[Int] and
      (JsPath \ "wpm").writeNullable[Double] and
      (JsPath \ "errorLen").writeNullable[Int] and
      (JsPath \ "potentialWpm").writeNullable[Double] and
      (JsPath \ "potentialLength").writeNullable[Int] and
      (JsPath \ "device").writeNullable[String] 
  )(unlift(TestSection.unapply))

  implicit val logdataSubmitReads: Reads[LogdataSubmit] = (
    (JsPath \ "etype").read[String] and
    (JsPath \ "key").read[String] and
    (JsPath \ "code").read[String] and 
    (JsPath \ "data").read[String] and 
    (JsPath \ "input").read[String] and
    (JsPath \ "pressed").read[String] and
    (JsPath \ "timestamp").read[Long] and
    (JsPath \ "screen_orientation").read[String] and
    (JsPath \ "device_orientation").read[String]
    )(LogdataSubmit.apply _)

  implicit val keystrokeSubmitReads: Reads[KeystrokeSubmit] = (
      (JsPath \ "pressTime").read[Long] and
      (JsPath \ "releaseTime").read[Long] and
      (JsPath \ "keycode").read[Int] and
      (JsPath \ "letter").read[String]
    )(KeystrokeSubmit.apply _)

  implicit val questionnaireDataSubmitReads: Reads[QuestionnaireDataSubmit] = (
      (JsPath \ "age").read[Int] and
      (JsPath \ "gender").read[String] and
      (JsPath \ "nativeLanguage").read[String] and
      (JsPath \ "hasTakenTypingCourse").read[String] and
      (JsPath \ "timeSpentTyping").read[String] and
      (JsPath \ "typeEnglish").read[String] and
      (JsPath \ "fingers").read[String] and 
      (JsPath \ "layout").read[String] and
      (JsPath \ "keyboardType").read[String] and
      (JsPath \ "usingApp").read[String] and
      (JsPath \ "usingFeatures").read[String] 
    )(QuestionnaireDataSubmit.apply _)
}
