"""
Functions to color clusters etc

Author: Luana Micallef, February 2016
"""

import math
import numpy as np
from itertools import combinations
from colormath.color_objects import LabColor, LCHabColor, sRGBColor
from colormath.color_conversions import convert_color
from colormath.color_diff import delta_e_cie2000 
import matplotlib.pyplot as plt # to plot and see rgb colors
import utilities
from PIL import Image



# Get n colors equidistant by hue angle in CIE LCH space
# (Munzner states that for categorical variables we should change the hue namely)
# (Splatterplots used 74.5 for L and 100 for C which is the color purity)
def getNHueEquidistantLCHColors(n):
    L = 50.0 #74.5
    C = 100.0   # c=chroma in [0..100]
    colors = [ LCHabColor(L,C,H) for H in np.linspace(0, 360, n, endpoint=False) ] 
    return colors



# Get 4 or 8 colors equidistant along the a and b color channels of LAB
# refer to the diagram in section "The CIE Lab Colour Space or Colour Model" of 
# http://www.colourphil.co.uk/lab_lch_colour_space.shtml
# Here L is constant as for RGB luminance is 50; however note that for CIELab
# the standard hue like standard red and yellow would not have same luminance
# and CIELab is the color space with luminance of colors close to that perceived
# BUT since we want all colors to have the same perceived luminance such they are all equally 
# visible with the white background then we set the perceived luminance to 50.0 that is daylight
# The ranges for a and b are theoretically infinite but the practical ranges are -127 to 128
# (see http://www.colourphil.co.uk/lab_lch_colour_space.shtml), but these ranges would be
# greater than those corresponding to RGB color space which through a simple calculation 
# are L in [0, 100], A in [-86.185, 98,254], B in [-107.863, 94.482]
# (src http://stackoverflow.com/questions/19099063/what-are-the-ranges-of-coordinates-in-the-cielab-color-space)
# yet when these ranges are used, same colors as those with ranges -127 to 128 are obtained
def get4or8HueEquidistantLABColors (n):
    
    a_b_values = np.array([[127,0],[-127,0],[0,127],[0,-127]])
    #a_b_values = np.array([[-86.185,0],[98.254,0],[0,-107.863],[0,94.482]]) # computed ranges to match rgb color space
   
    if (n==8):
        limits = np.array([127,-127])
        a_b_values2 = utilities.computeCartesianProductOfTwoNumpyArrays(limits,limits)
        a_b_values = np.concatenate((a_b_values,a_b_values2))
       
    L = 50 # 50 is daylight
    colors = [ LabColor(L,A,B) for A, B in a_b_values]
   
    return colors
    

# Returns 12 categorical distinct RGB colors in Lab form
# These colors were obtained from Ware 2012 pg126
def getCategoricalRGBColorsInLabForm():

    # Ware 2012 pg 126: "the 12 colors recommended for use in coding:
    # red, green, yellow, blue, black, white, pink, cyan, gray (128,128,128), orange (255,165,0),
    # brown (165,42,42), purple (128,0,128)"
    # Munzner 2014 pg 227 has magenta instead of gray
    # Since gray could be close to black with opacity then we used magenta instead of gray 
    # => red, green, yellow, blue, black, white, pink(255,192,203), cyan, magenta, 
    # orange (255,165,0), brown (165,42,42), purple (128,0,128)
    rgb_max = 255.0
    colors_rgb = [[255,0,0],[0,255,0],[255,255,0],[0,0,255],[0,0,0],[255,255,255], \
                  [255,192,203],[0,255,255],[255,0,255],[255,165,0],[165,42,42],[128,0,128]]
  
    colors_srgb = [sRGBColor(r/rgb_max,g/rgb_max,b/rgb_max) for r,g,b in colors_rgb]
    
    return convertRGBtoLabColors(colors_srgb)


# Returns the CIE lab of the colors of a highly saturated version of the colormap recommended by colorbrewer 
# for 8 categories; used the highly saturated to ensure that the colors are still distinguishable
# when the marker size is small as recommended also by Munzner 2014 and Ware 2012 for categories
# http://colorbrewer2.org  no. of data classes=8; nature of data: qualitative; 6th color scheme from the right
def getCategoricalColorBrewerColorsInLabForm():
    rgb_max = 255.0
    # 8 classes qualitative
    #colors_rgb = [[228,26,28],[55,126,184],[77,175,74],[152,78,163],[255,127,0],\
    #              [255,255,51],[166,86,40],[247,129,191]]
    # 5 classes 
    colors_rgb = [[228,26,28],[55,126,184],[77,175,74],[152,78,163],[255,127,0]]
    colors_srgb = [sRGBColor(r/rgb_max,g/rgb_max,b/rgb_max) for r,g,b in colors_rgb]
    return convertRGBtoLabColors(colors_srgb)
    

# Convert LCH color to CIE lab colors
def convertLCHtoLabColors(colors_lch):
    return [ convert_color(color_lch, LabColor) for color_lch in colors_lch ]

# Convert RGB colors to CIE lab colors
def convertRGBtoLabColors(colors_rgb):
    return [ convert_color(color_rgb, LabColor) for color_rgb in colors_rgb ]



# Get the visual difference (delta e equations) of each pair of n CIELab colors equidistant 
# by the hue angle in CIE LCH space 
def getDeltaEBetweenNDistinctLabColorsViaLCH (n):
    colors = convertLCHtoLabColors(getNHueEquidistantLCHColors(n))
    color_pairs = list(combinations(colors,2))
    deltas = {(c1,c2):delta_e_cie2000(c1,c2) for c1, c2 in color_pairs}
    return deltas
 
 
 
# Get the visual difference (delta e equations) of each pair of CIELab colors (at least n) 
# equidistant along the a and b color channels
def getDeltaEBetween4or8DistinctLabColors (n):
    colors = get4or8HueEquidistantLABColors(n)
    color_pairs = list(combinations(colors,2))
    deltas = {(c1,c2):delta_e_cie2000(c1,c2) for c1, c2 in color_pairs}
    return deltas


# Get the visual difference (delta e equations) of each pair of CIELab colors, 
# which colors are recommended by Ware 2012 and Munzner 2014 for categories
def getDeltaEBetweenCategoricalDistinctColors():
    colors = getCategoricalRGBColorsInLabForm()
    color_pairs = list(combinations(colors,2))
    deltas = {(c1,c2):delta_e_cie2000(c1,c2) for c1, c2 in color_pairs}
    return deltas    


# Get the visual difference (delta e equations) of each pair of CIELab colors, 
# which colors are a highly saturated version of the colormap recommended by colorbrewer 
# for 8 categories; used the highly saturated to ensure that the colors are still distinguishable
# when the marker size is small as recommended also by Munzner 2014 and Ware 2012 for categories
def getDeltaEBetweenCategoricalColorBrewerColors():
    colors = getCategoricalColorBrewerColorsInLabForm()
    color_pairs = list(combinations(colors,2))
    deltas = {(c1,c2):delta_e_cie2000(c1,c2) for c1, c2 in color_pairs}
    return deltas
    


# Returns an array for distinguishable LAB colors, one for each cluster 
# the closer or more overlap between two clusters, the more distinguishable their colors
# A greedy approach is adopted
def getLABColorsForClusters(pairwiseClusterOverlapMeasures, colorDeltas, TriangleIndices):
    
    overlapMeasures = {(pair[0], pair[1]): pairwiseClusterOverlapMeasures [pair[0],pair[1]] for pair in TriangleIndices} 
    overlapMeasures_keys_desc = sorted(overlapMeasures, key=overlapMeasures.__getitem__, reverse=True)
    
    color_percluster = np.array([None for i in pairwiseClusterOverlapMeasures[0]])
    colorDeltas_available = colorDeltas.copy()
   
    noOfClusters = color_percluster.size
    noOfColorsAssigned = 0
    colorsUsed = []
    maxdist = 0
    
    for key in overlapMeasures_keys_desc:
    
        # if no colors have been assigned to any of the clusters, 
        # assign the next 2 available colors that are most distinguishable and are still not assigned
        if ((color_percluster[key[0]] == None) and (color_percluster[key[1]] == None)):
            colorpair_options_keys_desc = sorted(colorDeltas_available, key=colorDeltas_available.get, reverse=True)
            chosen_colorpair = None
            for c1, c2 in colorpair_options_keys_desc:
                # check if the colors have been used already
                if (not(c1 in colorsUsed) and not(c2 in colorsUsed)):
                    chosen_colorpair = (c1,c2)
                    break 
            
            if (noOfColorsAssigned==0):
                maxdist = colorDeltas_available[chosen_colorpair]
                
            colorsUsed = colorsUsed + [chosen_colorpair[0],chosen_colorpair[1]]
            color_percluster[key[0]], color_percluster[key[1]] = chosen_colorpair
            colorDeltas_available.pop(chosen_colorpair, None)
            noOfColorsAssigned = noOfColorsAssigned + 2
           
           
        # if only one color has been assigned to a cluster c in pair of cluster,
        # assign the color with the highest visual difference from c to the other cluster and all other colors that have been assigned already
        elif ((color_percluster[key[0]] == None) or (color_percluster[key[1]] == None)):
            cluster_wcolor_indx = key[0] if (color_percluster[key[0]] != None) else key[1]
            cluster_wocolor_indx = key[0] if (color_percluster[key[0]] == None) else key[1]
            
            # find all the pairs of colors where one of the colors is the color assigned to one of the clusters being analyzed 
            # and sort them in descending order based on the visual difference between the 2 colors
            colorpair_options = {(c1,c2):v for (c1,c2),v in colorDeltas_available.items() if ((c1 == color_percluster[cluster_wcolor_indx]) or (c2 == color_percluster[cluster_wcolor_indx]))} 
            colorpair_options_keys_desc = sorted(colorpair_options, key=colorpair_options.get, reverse=True)
            
            # for each possible color option, make sure that 
            # the color is not already used/assigned and 
            # its perceptual distance/difference for all the other colors that have been assigned already is large enough
            first_colorpair = None
            chosen_colorpair = None
            chosen_cToFind = None
            for c1, c2 in colorpair_options_keys_desc:
                cKnown, cToFind = (c1, c2) if (c1 == color_percluster[cluster_wcolor_indx]) else (c2,c1)
                if (cToFind in colorsUsed):
                    continue
                mindist_cToFind_cUsed = None
                sumdists = colorDeltas[(c1,c2)]
                count = 1
                for cUsed in colorsUsed:
                    if ((cUsed == cKnown) or \
                        (not((cUsed,cToFind) in colorDeltas) and not((cToFind,cUsed) in colorDeltas)) ):
                        continue
                    dist_cToFind_cUsed = colorDeltas[(cToFind,cUsed)] if ((cToFind,cUsed) in colorDeltas) else colorDeltas[(cUsed,cToFind)]
                    if (mindist_cToFind_cUsed == None):
                        mindist_cToFind_cUsed = dist_cToFind_cUsed  
                        first_colorpair = (c1,c2) 
                    elif (mindist_cToFind_cUsed > dist_cToFind_cUsed): 
                        mindist_cToFind_cUsed = dist_cToFind_cUsed 
                    sumdists += dist_cToFind_cUsed
                    count += 1
                avgdist = sumdists/count  
                #if ((avgdist/maxdist) >= 0.5):
                if ((mindist_cToFind_cUsed/maxdist) >= 0.25):
                    chosen_colorpair = (c1,c2)
                    chosen_cToFind = cToFind
                    break
                    
            # if a color that is distance enough from all other assigned colors is not found, 
            # then use the first available color that has the largest difference from the color of the other cluster 
            if (chosen_cToFind == None):
                chosen_colorpair = first_colorpair
                chosen_cToFind = chosen_colorpair[1] if (chosen_colorpair[0] == color_percluster[cluster_wcolor_indx]) else chosen_colorpair[0]
                #print "(chosen_cToFind == None)"
                
            # save the color and update other relevant variables     
            color_percluster[cluster_wocolor_indx] = chosen_cToFind
            colorsUsed.append(color_percluster[cluster_wocolor_indx])            
            colorDeltas_available.pop(chosen_colorpair, None)
            noOfColorsAssigned = noOfColorsAssigned + 1
            
            
        # if both clusters have been assigned a colour already then 
        # delete the color pair from the available color list
        else:
            colorDeltas_available.pop((color_percluster[key[0]],color_percluster[key[1]]), None)
            colorDeltas_available.pop((color_percluster[key[1]],color_percluster[key[0]]), None)
            
        
        # check whether all clusters has been assigned a unique color
        if (noOfColorsAssigned >= noOfClusters):
            break
    
    # print color_percluster
    return color_percluster




# Converts an array of colors of form e.g., LAB or LCH to sRGBColor colors and then returns a numpy array 
# of the form [(r,g,b), ...] where each of r, g and b values are in [0,1] if clamp = True
def convertColorsToRGBColors (colors_lab, clamp=False):

    colors_rgb = []
    for c in colors_lab:
        c_srgb = convert_color(c,sRGBColor)
        c_rgb = (c_srgb.clamped_rgb_r, c_srgb.clamped_rgb_g, c_srgb.clamped_rgb_b) if (clamp==True) \
                    else (c_srgb.rgb_r, c_srgb.rgb_g, c_srgb.rgb_b) 
        colors_rgb.append(c_rgb) 
    
    return np.array(colors_rgb)
    
    

# Returns an array for distinguishable RGB colors (each color of the form (r,g,b) 
# where each of r, g and b values is in [0,1])), one for each cluster 
# the closer or more overlap between two clusters, the more distinguishable their colors
def getRGBColorsForClusters(pairwiseClusterOverlapMeasures, colorDeltas, TriangleIndices):
    
    # if 1 cluster return color black
    if (pairwiseClusterOverlapMeasures.shape == (1,1)):
        return np.array([[0.0, 0.0, 0.0]])

    color_percluster_lab = getLABColorsForClusters(pairwiseClusterOverlapMeasures, colorDeltas, TriangleIndices)
    return convertColorsToRGBColors(color_percluster_lab, clamp=True) 
    



# Plot dots with rgb colors to see what the color looks like
# input: numpy array of the (r,g,b) of all colors e.g., [[r1,g1,b1],[r2,g2,b2],...]
# For testing namely
def plotRGBColors (rgb_colors):

    marker_size = 20
    pnts = np.array([[i,10] for i in range(0,rgb_colors.size)])
    for [x,y],c in zip(pnts,rgb_colors):
        plt.plot(x, y, 'o', markersize=marker_size, color=c)
    plt.show()
    

# Plot dots with colors of form e.g., LAB or LCH to see what the color looks like
# input: numpy array of LabColor objects
# For testing namely
def plotColors (colors):
    plotRGBColors(convertColorsToRGBColors(colors, clamp=True))
  



# Checks if the RGB component for an RGBA is as required
# e.g., if rgba=(0,0,0,0) and rgb_required=(0,0,0) then isRgbOfRgbaAsRequired=True 
#       if rgba=(255,255,255,0) and rgb_required=(0,0,0) then isRgbOfRgbaAsRequired=False
def isRgbOfRgbaAsRequired(rgba, rgb_required):
    r, g, b, a = rgba 
    r_r, g_r, b_r =  rgb_required
    return ((r==r_r) and (g==g_r) and (b==b_r)) 



# Check if a pixel defined in RGBA is white
# RGBAs for white: (255,255,255,255) -> white, fully opaque; 
#                  (255,255,255,0) -> white fully transparent;
#                  (0,0,0,0) -> black fully transparent
def isRgbaWhite(rgba): 
    whiteRGBAs = [(255,255,255,255), (255,255,255,0), (0,0,0,0)]
    return (rgba in whiteRGBAs) 
    


# Returns an RGBA for white that has the RGB component black 
# i.e., black with full transparency (opacity=0) -> (0,0,0,0)
def getWhiteRgbaWithBlackRgb(): 
    return (0,0,0,0)    
  
  

# Check if the inputted rgba for a gray color is defined only by its alpha 
# (a black with an opacity o i.e., (0,0,0,o)); if yes just return the rgba, 
# if not and the rgba is white, return white rgba in the correct form,
# if not and the rgba is not white, return None
def defineGrayRgbaByAlphaOnly(rgba):
    blackRGB = (0,0,0)
    if (not isRgbOfRgbaAsRequired(rgba, blackRGB)):
        if (isRgbaWhite(rgba)):
            return getWhiteRgbaWithBlackRgb()
        else: 
            print "ERROR: Cannot define the gray value of RGBA " + rgba + " by only alpha"
            return None 
    return rgba



# Get normalized gray-level value of RGBA
# i.e., the alpha of the RGBA value
# normalized by /255 -> assuming that each value in rgba is in [0,255] not [0,1]
def getNormalizedGrayValueForRGBA(rgba):
    rgba = defineGrayRgbaByAlphaOnly(rgba)
    if (rgba == None):
        return None
    red, green, blue, alpha = rgba
    return scaleRGBComponent0to1(alpha)
    

# Return the relative luminance value of an RGBA color in the form of (r,g,b,a)
# e.g., if rgba = (255,255,255,255) then relLuminance = 255.0
def getRelativeLuminanceOfRGBA_fromAlpha (rgba):
    rgba = defineGrayRgbaByAlphaOnly(rgba)
    if (rgba == None):
        return None
    red, green, blue, alpha = rgba
    return 1-scaleRGBComponent0to1(alpha)
    


# Return the relative luminance value of an RGB color in the form of (r,g,b)
# e.g., if rgb = (255,255,255) then relLuminance = 255.0
def getRelativeLuminanceOfRGB (rgb):
    red, green, blue = rgb
    relLuminance = (.299 * red) + (.587 * green) + (.114 * blue)
    return relLuminance


# Convert an RGB from 0 to 255 to 0 to 1
def scaleRGBComponent0to1(rgb):
    return float(rgb)/255.0


    
    
    
# MAIN
# See the colors we have 
#plotColors(get4or8HueEquidistantLABColors(8))
#plotColors(getNHueEquidistantLCHColors(15))
#plotColors(getCategoricalRGBColorsInLabForm(#))







