#include <ax12.h>
// #include "Filter.h"


// CONSTANTS
#define LED_PIN                5
#define BUTTON_PIN             2
#define FINGER_POT_PIN         0
#define POT_PIN                0
#define FSR_PIN                1
#define MOTOR_ID_UPPER         1
#define MOTOR_ID_LOWER         2

#define BUFFER_SIZE            8
#define INPUT_SIZE             120
#define DELAY_LONG             30
#define DELAY_SHORT            1
#define DELAY_TX               2
#define SERVO_MIN              0
#define SERVO_MAX              1023
#define MAX_TORQUE             1023

// Actions
#define EXECUTE_SEQUENCE       2
#define RANDOM_ACTION          1
#define SETUP                  6
#define TEST_CONNECTION        7
#define SET_HIGHT              3

// robot parameters
const float rJoint = 0.005;
const float rTip = 0.095;
const float staticFriction = 1.6;

// coefficient to convert angle to arc
const float angle2arcf = (PI / 180) * rJoint;
const float units2deg = 0.06841;

// muscle model constants
const float PCSA=4;
const float ks=0.8*PCSA;
const float kc=6;
const float kd=0.1*ks;
const float mlength0=0.3;   // initial length of the muscle (m)
const float baseStrech=0.03;

// GLOBAL VARIABLES
float fingerAngle = 0;
float angVelocity = 0;
float lastAngle = 0;
float targetForceLower = 0;
float targetForceUpper = 0;
int fingerRawPos;
int fsrReading;
int press_n = 0;
int randomActionDuration = 0;
float peakForce;
float impulse;
unsigned int timeStep;
float table[INPUT_SIZE][3] = {0.0};
byte command = SETUP;
int currentInputSize = INPUT_SIZE;
float debug_logging;
float debug_logging2;


/* limits (initial values obtained empirically)
Upper servo: max: 709, min: 582, d: 127
Lower servo: max: 849, min: 713, d: 136
Finger:      max: 127, min: 666, d: -539 */
int limitUpperMax = 709;
int limitUpperMin = 582;
int limitLowerMax = 849;
int limitLowerMin = 713;
int limitFingerMax = 127;
int limitFingerMin = 666;

// fingers default position (muscles relaxed)
float angleOffset = 16.84;  // deg
int unitOffset = round((8.745 + angleOffset) / units2deg);


// filter class
class RunningAverage {
  public:
    void addReading(float value) {
      total = total - readings[readIndex];
      readings[readIndex] = value;
      total = total + readings[readIndex];
      readIndex++;
      if (readIndex >= numReadings) readIndex = 0;
    }

    float current() {
      return total / numReadings;
    }

    void reset(float fill) {
      memset(readings, fill, sizeof(readings));
      total = numReadings * fill;
    }

  private:
    static const int numReadings = 3;
    float readings[numReadings] = {0.0};
    int readIndex = 0;
    float total = 0;
};

RunningAverage velFilter;


// PROGRAM
void setup() {
  ax12Init(1000000);
  Serial.begin(115200);
  pinMode(BUTTON_PIN, INPUT_PULLUP);
  pinMode(LED_PIN, OUTPUT);
  // Servo settings: enabled maximum torque
  resetServos();
}

void loop() {
  bool newCommand = readCommand();
  if (newCommand) {
    newCommand = executeCommand();
  }
  delay(DELAY_LONG);
}


// return true if command was read, otherwise false
bool readCommand() {
  if (Serial.available() > 0) {
    command = Serial.read();
    parseCommand();
    return true;
  }
  return false;
}


void parseCommand() {
  if (command == TEST_CONNECTION) {
    Serial.print("Ready!");
  }
  if (command == EXECUTE_SEQUENCE) {
    // read the input into a table
    timeStep = readInt();
    currentInputSize = readInt();
    for (int i = 0; i < currentInputSize; i++) {
      for (int j = 0; j < 2; j++) {
        table[i][j] = readFloat();
      }
      Serial.write(1);
    }
  }
  if (command == RANDOM_ACTION) {
    // read duration of random actions
    randomActionDuration = readInt();
  }
  if (command == SET_HIGHT) {
    angleOffset = readFloat();
  }
}


bool executeCommand() {
  if (command == EXECUTE_SEQUENCE) {
    executeSequence();
  }
  if (command == SETUP) {
    executeSetup();
  }
  if (command == RANDOM_ACTION) {
    executeRandomActions();
  }
  if (command == SET_HIGHT) {
    // offset for the rest position in raw units
    unitOffset = round((8.745 + angleOffset) / units2deg);
    writeInt(unitOffset);
  }
  return false;
}


void executeRandomActions() {
  unsigned long N = (unsigned long) ((float) randomActionDuration * 1000.0 / 10.0);
  for (unsigned long i; i < N; i++) {
    int upper = random(limitUpperMin, limitUpperMax);
    int lower = random(limitLowerMin, limitLowerMax);
    setRegister(MOTOR_ID_UPPER, AX_GOAL_POSITION_L, upper);
    setRegister(MOTOR_ID_LOWER, AX_GOAL_POSITION_L, lower);
    delay(10);
  }
  writeInt(1);
}


void executeSetup() {
  // release torque, for manual configuring
  setRegister1(MOTOR_ID_UPPER, AX_TORQUE_ENABLE, false);
  setRegister1(MOTOR_ID_LOWER, AX_TORQUE_ENABLE, false);

  // turn on the upper servo ligth
  setRegister1(MOTOR_ID_UPPER, AX_LED, true);
  // manually configure to upper position
  waitForButton();
  // press button to record both servo positions limits for upper position
  limitUpperMax = getRegister(MOTOR_ID_UPPER, AX_PRESENT_POSITION_L);
  limitLowerMin = getRegister(MOTOR_ID_LOWER, AX_PRESENT_POSITION_L);
  // limitFingerMax = 1023 - analogRead(FINGER_POT_PIN);
  limitFingerMax = analogRead(FINGER_POT_PIN);

  // turn off the upper servo ligth and turn on lower led
  setRegister1(MOTOR_ID_UPPER, AX_LED, false);
  setRegister1(MOTOR_ID_LOWER, AX_LED, true);

  // manually configure to upper position
  waitForButton();
  // press button to record both servo positions limits for lower position
  limitUpperMin = getRegister(MOTOR_ID_UPPER, AX_PRESENT_POSITION_L);
  limitLowerMax = getRegister(MOTOR_ID_LOWER, AX_PRESENT_POSITION_L);
  // limitFingerMin = 1023 - analogRead(FINGER_POT_PIN);
  limitFingerMin = analogRead(FINGER_POT_PIN);
  setRegister1(MOTOR_ID_LOWER, AX_LED, false);

  // turn torque back on
  setRegister1(MOTOR_ID_UPPER, AX_TORQUE_ENABLE, true);
  setRegister1(MOTOR_ID_LOWER, AX_TORQUE_ENABLE, true);

  // set offsetCenter
  // offsetCenter = (int) ((float)(limitFingerMax + limitFingerMin)/2.0);

  // report values
  writeInt(limitUpperMax);
  writeInt(limitLowerMax);
  writeInt(limitFingerMax);
  writeInt(limitUpperMin);
  writeInt(limitLowerMin);
  writeInt(limitFingerMin);
}



void executeSequence() {
  resetServos();
  velFilter.reset(0.0);
  delay(200); // wait so that position can be reached

  // Change settings for torque control
  setRegister1(MOTOR_ID_UPPER, AX_CW_COMPLIANCE_SLOPE, 2);
  setRegister1(MOTOR_ID_UPPER, AX_CCW_COMPLIANCE_SLOPE, 2);
  setRegister(MOTOR_ID_UPPER, AX_TORQUE_LIMIT_L, 0);
  setRegister(MOTOR_ID_LOWER, AX_TORQUE_LIMIT_L, 0);
  setRegister(MOTOR_ID_UPPER, AX_GOAL_POSITION_L, limitUpperMax);
  setRegister(MOTOR_ID_LOWER, AX_GOAL_POSITION_L, limitLowerMax);

  bool first = true;
  int fsrOffset = analogRead(FSR_PIN);
  press_n = 0;
  impulse = 0.0;
  int index = 0;
  unsigned long dt = 0;
  unsigned long totalTime = 0;
  unsigned long endTime = timeStep * currentInputSize;
  giveLedSignal();
  unsigned long currentTime;
  unsigned long startTime = millis();
  unsigned long lastTime = startTime;
  // loop until input index or time limit is reached
  while ((totalTime < endTime) && (index < currentInputSize)) {
    currentTime = millis();
    dt = currentTime - lastTime;
    totalTime = currentTime-startTime;

    if (totalTime >= (index+1)*timeStep-dt) {
      readSensoryData(index, dt, totalTime, &first);
      updateMuscle(index, dt);
      updateTable(index, dt+totalTime, fingerAngle, fsrReading, fsrOffset);
      // updatePeakForce();
      updateImpulse(dt);
      updateServos(index, fsrReading, fsrOffset);
      index++;
    }

    lastTime = currentTime;
    lastAngle = fingerAngle;
  }
  giveLedSignal();
  setRegister(MOTOR_ID_UPPER, AX_TORQUE_LIMIT_L, 0);
  setRegister(MOTOR_ID_LOWER, AX_TORQUE_LIMIT_L, 0);
  // Serial.println();
  writeTable();
  writeInt(press_n);
  writeFloat(impulse);
}


void giveLedSignal() {
  digitalWrite(LED_PIN, true);
  delay(500);
  digitalWrite(LED_PIN, false);
}

void resetServos() {
  // change settings so that start position can be reached smoothly
  setRegister1(MOTOR_ID_UPPER, AX_CW_COMPLIANCE_SLOPE, 0x40);
  setRegister1(MOTOR_ID_UPPER, AX_CCW_COMPLIANCE_SLOPE, 0x40);
  setRegister(MOTOR_ID_UPPER, AX_TORQUE_LIMIT_L, 1023);
  setRegister(MOTOR_ID_LOWER, AX_TORQUE_LIMIT_L, 1023);

  // get to start position
  int restPosUpper = map(unitOffset, limitFingerMin, limitFingerMax,
                         limitUpperMin, limitUpperMax);
  int restPosLower = map(unitOffset, limitFingerMin, limitFingerMax,
                         limitLowerMax, limitLowerMin);
  setRegister(MOTOR_ID_UPPER, AX_GOAL_POSITION_L, restPosUpper);
  setRegister(MOTOR_ID_LOWER, AX_GOAL_POSITION_L, restPosLower);
}


void recordPress(unsigned long totalTime) {
  // mid air button hack
  // bool currentStatus = fingerAngle > 2.6;

  // real button
  bool currentStatus = !digitalRead(BUTTON_PIN);

  // indicator led
  digitalWrite(LED_PIN, currentStatus);

  // record the time the button was pressed first
  if (press_n == 0 && currentStatus) press_n = totalTime;
}


void readSensoryData(int index, unsigned long dt, unsigned long totalTime,
                     bool* first) {
  fingerAngle = getFingerAngle();
  if (*first) {
    lastAngle = fingerAngle;
    *first = false;
  }
  angVelocity = (fingerAngle - lastAngle) / dt;
  velFilter.addReading(angVelocity);
  fsrReading = analogRead(FSR_PIN);
  // press_n += !digitalRead(BUTTON_PIN);
  recordPress(totalTime);
}


void updateMuscle(int index, unsigned long dt) {
  static float mlengthUpper = 0;
  static float xmlengthUpper = 0;
  static float mlengthRateUpper = 0;
  static float mlengthLower = 0;
  static float xmlengthLower = 0;
  static float mlengthRateLower = 0;
  static float strainUpper = 0;
  static float xstrainUpper = 0;
  static float strainRateUpper = 0;
  static float strainLower = 0;
  static float xstrainLower = 0;
  static float strainRateLower = 0;
  mlengthUpper = getUpperMuscleLength(fingerAngle);
  mlengthRateUpper = (mlengthUpper-xmlengthUpper) / dt;
  strainUpper = getStrain(mlengthUpper);
  strainRateUpper = (strainUpper-xstrainUpper) / dt;
  mlengthLower = getLowerMuscleLength(fingerAngle);
  mlengthRateLower = (mlengthLower-xmlengthLower)/dt;
  strainLower = getStrain(mlengthLower);
  strainRateLower = (strainLower-xstrainLower) / dt;
  xmlengthUpper = mlengthUpper;
  xmlengthLower = mlengthLower;
  xstrainUpper = strainUpper;
  xstrainLower = strainLower;

  targetForceUpper = muscle(table[index][0], strainUpper, strainRateUpper,
                            mlengthRateUpper, mlengthUpper);
  targetForceLower = muscle(table[index][1], strainLower, strainRateLower,
                            mlengthRateLower, mlengthLower);
}


void updateServos(int index, int fsrReading, int fsrOffset) {
  float tolerance = 0.04;    // units in Newtons
  float dForce = targetForceUpper - targetForceLower;
  float velocity = velFilter.current();
  float touchForce = (93.0 / 19.0) * convertFsrToForce(fsrReading, fsrOffset);

  // compensate for friction in the servomotor if the servomotors are not
  // supposed to be countering each other
  if (dForce < -tolerance) {
    // No need to compensate when solid object blocking and (servo is assumed
    // to be static static)
    // float eps = 0.1;
    // if (!((-eps < velocity && velocity < eps) &&
    //     ((93.0 / 19.0) * touchForce) >= -dForce)) {
    if (!(((93.0 / 19.0) * touchForce) >= -dForce)) {
      targetForceLower = compensatedForce(targetForceLower,
                                          max(0, velocity));
    }
  } else if (dForce > tolerance) {
    targetForceUpper = compensatedForce(targetForceUpper,
                                        max(0, -velocity));
  } 


  // convert to units for servocommand
  int upper = force2servoUnits(targetForceUpper);
  int lower = force2servoUnits(targetForceLower);

  // when out of bounds the max torque is set to 0
  if (fingerRawPos < limitFingerMax) {
    setRegister(MOTOR_ID_UPPER, AX_TORQUE_LIMIT_L, 0);
  } else {
    setRegister(MOTOR_ID_UPPER, AX_TORQUE_LIMIT_L, upper);
  }
  if (fingerRawPos > limitFingerMin) {
    setRegister(MOTOR_ID_LOWER, AX_TORQUE_LIMIT_L, 0);
  } else {
    setRegister(MOTOR_ID_LOWER, AX_TORQUE_LIMIT_L, lower);
  }
}


float getFingerAngle() {
  // posFilter.Filter(analogRead(FINGER_POT_PIN));
  // fingerRawPos = 1023 - posFilter.Current();

  // float raw_angle = analogRead(FINGER_POT_PIN);
  // float angle = 0.06841 * raw_angle - 8.745;
  fingerRawPos = analogRead(FINGER_POT_PIN);
  // return ((float)(fingerRawPos - offsetCenter)) * units2deg;
  return units2deg * fingerRawPos - 8.745 - angleOffset;
}


float getUpperMuscleLength(float angle) {
  float len = max(0.0, mlength0 + baseStrech + angle * angle2arcf);
  return len;
}


float getLowerMuscleLength(float angle) {
  float len = max(0, mlength0 + baseStrech - angle * angle2arcf);
  return len;
}


float getStrain(float mlength) {
  return (mlength-mlength0)/mlength0;
}


float muscle(float activation, float strain, float strain_rate,
             float mlength_v, float mlength)
{
  float kmax=PCSA/(mlength-mlength0);
  float lm=0.5*mlength0;
  float vm=8.0*mlength0;
  float fp=max(0.0,ks*(exp(kc*strain)-1)+kd*strain_rate);
  float force;
  float Fl=max(0.0,kmax*(mlength-lm));
  float Fv=max(0,1+min(mlength_v,0.0)/vm);
  float fc=activation*Fl*Fv;
  force=fp+fc;
  return force;
}

void updatePeakForce() {
  float current = abs(targetForceLower - targetForceUpper);
  if (peakForce < current) {
    peakForce = current;
  } 
}

void updateImpulse(unsigned long dt) {
  float currentTotalForce = targetForceLower + targetForceUpper;
  impulse += currentTotalForce*(float)dt;
}



int limitUpper(int value) {
  // return constrain(value, limitUpperMin, limitUpperMax);
  return constrain(value, SERVO_MIN, SERVO_MAX);
}


int limitLower(int value) {
  // return constrain(value, limitLowerMin, limitLowerMax);
  return constrain(value, SERVO_MIN, SERVO_MAX);
}


void updateTable(int i, unsigned long time, float angle, int fsr,
                 int fsrOffset) {
  table[i][0] = (float) time;
  table[i][1] = rTip * angle * PI / 180;
  table[i][2] = convertFsrToForce(fsr, fsrOffset);
}


float convertFsrToForce(int fsrRaw, int offset) {
  // convert raw fsr sensor value to force in newtons.
  float force = 0.25*exp(0.0063 * (float) fsrRaw)-0.25;
  float force_offset = 0.25*exp(0.0063 * (float) offset)-0.25;
  return force-force_offset;
}


void writeTable() {
  for (int i = 0; i < currentInputSize; i++) {
    for (int j = 0; j < 3; j++) {
      writeFloat(table[i][j]);
    }
    readInt();
  }
}


int force2servoUnits(float force) {
  int units = round(15.401 * force + 32.246);
  return constrain(units, 0, 1023);
}


float compensatedForce(float force, float velocity) {
  float friction = (402.16 * velocity + sign(velocity) * 6.0196);
  // float friction = sign(velocity) * 6.0196;
  return max(force, force + friction);
}


float sign(float value) {
  if (value < 0) return -1;
  if (value > 0) return 1;
  return 0;
}


void writeInt(int value) {
  // while(!(Serial.availableForWrite()>=2));
  byte buffer[BUFFER_SIZE];
  memcpy(buffer, &value, 2);
  Serial.write(buffer, 2);
}


void writeFloat(float value) {
  // while(!(Serial.availableForWrite()>=4));
  byte buffer[BUFFER_SIZE];
  memcpy(buffer, &value, 4);
  Serial.write(buffer, 4);
}


int readInt() {
  while (Serial.available() < 2);
  byte buffer[2];
  int output;
  buffer[0] = Serial.read();
  buffer[1] = Serial.read();
  memcpy(&output, &buffer, 2);
  return output;
}


float readFloat()  {
  while (Serial.available() < 4);
  byte buffer[4];
  float output;
  for (int i = 0; i < 4; i++) {
    buffer[i] = Serial.read();
  }
  memcpy(&output, &buffer, 4);
  return output;
}


bool waitForButton() {
  bool button = !digitalRead(BUTTON_PIN);
  bool xbutton = button;
  while (!(button && !xbutton)) {
    xbutton = button;
    button = !digitalRead(BUTTON_PIN);
    delay(DELAY_SHORT);
  }
  return true;
}


int setRegister(int id, int reg, int value) {
  delay(DELAY_TX);
  ax12SetRegister2(id, reg, value);
  return 1;
}


int getRegister(int id, int reg) {
  delay(DELAY_TX);
  return ax12GetRegister(id, reg, 2);
}


int setRegister1(int id, int reg, int value) {
  delay(DELAY_TX);
  ax12SetRegister(id, reg, value);
  return 1;
}


int getRegister1(int id, int reg) {
  delay(DELAY_TX);
  return ax12GetRegister(id, reg, 1);
}
